#!/usr/bin/python3
# coding: utf-8

import os
import struct, sys, time
import configparser
import math

exponentGlobal = []
nameGlobal = ""
typeGlobal = ""
primaryAddress = 0      # Primary address.
mbusini = configparser.RawConfigParser()
ret = mbusini.read("./MbusOPC.ini" ,encoding="ISO-8859-1")

# List of days in week
DaysInWeek = {
    0x00:"Not specified", 0x01:"Monday", 0x02:"Tuesday", 0x03:"Wednesday",
    0x04:"Thursday", 0x05:"Friday", 0x06:"Saturday", 0x07:"Sunday"
}

def expToInt(str):
    if len(str) > 0 and str[0].lower() == 'e':
        exp = int(str[1:])
    else:
        exp = 0
    return exp

def human_value(value, exponent):
    if exponent == None:
        return value
    if exponent >0:
        s = value * int(pow(10,exponent))
    elif exponent <0:
        decimals = exponent * -1
        div = int(pow(10,decimals))
        integer = int(value)/div
        if(value > 0):
            frac = int(value)%div
        else:
            frac = (int(value)%(div*-1))*-1
        s = "%d.%0*d" % (integer, decimals, frac)
    else:
        s = value

    return s

# Combinable (orthogonal) VIFE-codes.
def getCombinableVIFE(vife, section, physicalQuantity, unit, exponent):
    try:
        a=mbusini.get(section, "%02X"%(vife&0x7F))
        a=a.split(',')
        if a[0] != "":
            if exponent == None:
                exponent = expToInt(a[0])
            else:
                exponent += expToInt(a[0])
        if a[1] != "":
            unit += " " + a[1]
        if a[3] != "":
            physicalQuantity += " " + a[3]
    except:
        pass
    return physicalQuantity, unit, exponent

# Plain-text VIF.
def getPlainASCII(length, asci):
    asci = bytearray(asci)
    asci.reverse()
    #print("ascii reverse", asci)
    retval = "".join(chr(i) for i in asci)
    retval = retval.replace("\""," ").replace("\'"," ")
    return retval    
    # Kolla Elvaco prim.adr 19.

# Plain-text VIF. Replacing dots and spaces.
def getPlainASCII_1(length, asci):
    asci = bytearray(asci)
    asci.reverse()
    #print("ascii reverse", ascii)
    retval = "".join(chr(i) for i in asci)
    retval = retval.replace(". ","_").replace(".","_").replace(" ","_")
    return retval
    # Kolla Elvaco prim.adr 19.

# Third extension VIFE-codes.
def getVIFE_EF(vife):
    #print("VIFE = %02X"%(vife))
    # Reserved for future use. Not implemented in EN13757-3.
    physicalQuantity = "Not implemented VIFE_EF %02X" % vife
    return physicalQuantity, "", None, False, "VIFE_EF"

def getVIFE_FF(vife,  opcname):
    #print("VIFE = %02X"%(vife))
    physicalQuantity = "Unknown VIFE_FF %02X for manufacturer %s" % (vife, opcname)
    unit = ""
    templateVIB = "VIFE_FF"
    exponent = None
    try:
        a=mbusini.get(opcname, "%02X"%(vife&0x7F))
        a=a.split(',')
        if len(a) > 0:
            if bool(a[2]) and  len(a[0])>0:
                exponent = expToInt(a[0])
            unit = a[1]
            physicalQuantity = a[3]
            templateVIB=a[5]
    except:
        pass
    isDateTime = False
    return physicalQuantity, unit, exponent, isDateTime,  templateVIB

# Second extension VIFE-codes.
def getVIFE_FD(vife):
    #print("VIFE = %02X"%(vife))
    unit = "Not implemented VIFE FD %02X" %(vife)
    physicalQuantity = ""
    exponent = None
    templateVIB = ""
    try: #If this fails, something is missing from MBus opc.ini
        a=mbusini.get("VIFE_FD", "%02X"%(vife&0x7F))
        a=a.split(',')
        if len(a) > 0:
            if bool(a[2]) and  len(a[0])>0:
                exponent = expToInt(a[0])
            unit = a[1]
            physicalQuantity = a[3]
            templateVIB=a[5]
    except:
        pass
    isDateTime = False
    return physicalQuantity, unit, exponent, isDateTime, templateVIB

# First extension VIFE-codes.
def getVIFE_FB(vife):
    #print("VIFE = %02X"%(vife))
    unit = "Not implemented VIFE FB %02X" % (vife)
    physicalQuantity = ""
    exponent = None
    a=mbusini.get("VIFE_FB", "%02X"%(vife&0x7F))
    a=a.split(',')
    templateVIB=""
    try: #If this fails, something is missing from MBus opc.ini
        if len(a) > 0:
            if len(a[0])>0:
                exponent = expToInt(a[0])
            unit = a[1]
            physicalQuantity = a[3]
            templateVIB=a[5]
    except:
        pass
    isDateTime = False
    return physicalQuantity, unit, exponent, isDateTime,  templateVIB

# Primary VIF-codes.
def getVIF(vif):
    #print("VIF = %02X"%(vif))
    unit = "Not implemented VIF %02X" % (vif)
    templateVIB= "VIF%02X"%vif
    physicalQuantity = ""
    exponent = None
    isDateTime = False
    if (vif & 0x7F) == 0x6c or (vif & 0x7F) == 0x6d:
        isDateTime = True
    a=mbusini.get("VIF", "%02X"%(vif&0x7F))
    a=a.split(',')
    if len(a) > 0:
        if len(a[0])>0:
            exponent = expToInt(a[0])
        unit = a[1]
        physicalQuantity = a[3]
        templateVIB=a[5]
    return physicalQuantity, unit, exponent, isDateTime,  templateVIB

def printhex(data):
    tstr = ""
    for j in range(len(data)):
        if j%16 == 0 and j != 0:
            tstr += "\n"
        tstr += "%02X "%data[j]
    print(tstr)
    print("")

def bcdtoval(arr):
    # Kolla: Prim.Adr 4, objekt 35 innehåller 0xFF. Hur ska det tolkas?
    retval=0
    mul=1
    if (arr[-1] & 0xf0) == 0xf0:
        arr[-1] = arr[-1] & 0x0f
        Neg = True
    else:
        Neg = False
    for i in arr:
        j = i
        retval = retval+(j&0x0f)*mul
        mul = mul * 10
        retval = retval+((j&0xf0)>>4)*mul
        mul = mul * 10
    if Neg:
        retval *= -1
    return retval


# Decode CP16 from INT16 (Type G) to YY-MM-DD.
def DecodeCP16(value):
    # Prim.adr 42 objekt 4. Prim.adr 43 objekt 4, 9, 11.
    if value == -1:
        return "Invalid"
    y1 = (value & 0xF000) >> 8
    y2 = value & 0x00E0
    mon = (value & 0x0F00) >> 8
    day = value & 0x001F
    year = (y1 >> 1) | (y2 >> 5)
    if day == 0:
        return "Every day"
    elif mon == 15:
        return "Every month"
    elif year == 127:
        return "Every year"

    if year <= 80:
        year += 2000
    else:
        year += 1900
    return "%04d-%02d-%02d"%(year, mon, day)

# Decode CP24 from INT24 (Type J) to HH:MM:SS.
def DecodeCP24(value):
    sec = value & 0x00003F
    min = (value & 0x003F00) >> 8
    hour = (value & 0x1F0000) >> 16
    if sec == 63:
        return "Every second"
    elif min == 63:
        return "Every minute"
    elif hour == 31:
        return "Every hour"
    return "%02d:%02d:%02d"%(hour, min, sec)

# Decode CP32 from INT32 (Type F) to YY-MM-DD HH:MM.
def DecodeCP32(value):
    if value == -1:
        return "Invalid"

    # Prim.adr 25, objekt 20.
    min  =  value & 0x0000003F
    hour = (value & 0x00001F00) >> 8
    day  = (value & 0x001F0000) >> 16
    mon  = (value & 0x0F000000) >> 24
    y1   = (value & 0xF0000000) >> 24
    y2   = (value & 0x00E00000) >> 16
    hy   = (value & 0x00006000) >> 13 # Hundred years
    iv   = (value & 0x00000080) >> 7
    if iv == 1:
        return "Invalid"
    year = (y1 >> 1) | (y2 >> 5)

    if min == 63:
        return "Every minute"
    elif hour == 31:
        return "Every hour"
    elif day == 0:
        return "Every day"
    elif mon == 15:
        return "Every month"
    elif year == 127:
        return "Every year"

    if hy == 0 and year <= 80:
        year += 2000
    else:
        year += hy * 100 + 1900
    return "%04d-%02d-%02d %02d:%02d"%(year, mon, day, hour, min)

# Decode CP48 from INT48 (Type I) to YY-MM-DD HH:MM:SS, day, week, period.
def DecodeCP48(value):
    #print("JJJJJ")
    '''
         48    41   40    33   32    25   24    17   16     9   8      1
         47    40   39    32   31    24   23    16   15     8   7      0
    DT = 00000000 | 00000000 | 00000000 | 00000000 | 00000000 | 00000000
         --______   ----____   ---_____   ---_____   -_------   _-______
         ds Week	 y1 mon     y2 day    dow hour   ih  min    lt  sec
    '''
    #ds 	    = (value & 0xC00000000000) >> 40   # Daylight saving deviation (hour).
    week    = (value & 0x3F0000000000) >> 40
    y1	    = (value & 0x00F000000000) >> 33
    mon	    = (value & 0x000F00000000) >> 32
    y2	    = (value & 0x0000E0000000) >> 29
    day	    = (value & 0x00001F000000) >> 24
    dow	    = (value & 0x000000E00000) >> 21   # Day of week.
    hour	= (value & 0x0000001F0000) >> 16
    iv	    = (value & 0x000000008000) >> 15   # Time invalid.
    #h	    = (value & 0x000000004000) >> 8    # Daylight saving deviation (hour).
    min	    = (value & 0x000000003F00) >> 8
    #l	    = (value & 0x000000000080)         # Leap year.
    per     = (value & 0x000000000040)         # Time during daylight savings, period.
    sec	    = (value & 0x00000000003F)

    if iv == 1:
        return "Invalid"
    year = y1 + y2
    if sec == 63:
        return "Every second"
    elif min == 63:
        return "Every minute"
    elif hour == 31:
        return "Every hour"
    elif day == 0:
        return "Every day"
    elif mon == 15:
        return "Every month"
    elif year == 127:
        return "Every year"
    elif week == 0:
        return "Every week"

    if year <= 80:
        year += 2000
    else:
        year += 1900
    dayInWeek = DaysInWeek[dow]                         # Day in week.
    period = "winter"
    if per != 0x00:
        period = "summer"   # Winter or summer time period.

    strng =  "%04d-%02d-%02d %02d:%02d:%02d %s w%02d %s"%(year, mon, day, hour, min, sec, dayInWeek, week, period)
    return strng

# Decode a BCD to DateTime. Unsupported coding of date and time, but is common for ABB and BMeter.
def DecodeBCD12(value):
    dateTime = "%012d"%(value)
    year = dateTime[0:2]
    mon =  dateTime[2:4]
    day =  dateTime[4:6]
    hour = dateTime[6:8]
    min =  dateTime[8:10]
    sec =  dateTime[10:12]
    return "%s-%s-%s %s:%s:%s"%(year, mon, day, hour, min, sec)

# Decode CP16, CP24, CP32 and CP48.
def DecodeDateTime(dataType, value):
    if dataType == 0x02: # INT16 = CP16.
        #vib = "Date YYYY-MM-DD"
        return DecodeCP16(value)
    elif dataType == 0x03: # INT24 = CP24.
        #vib = "Time HH:MM:SS"
        return DecodeCP24(value)
    elif dataType == 0x04: # INT32 = CP32.
        #vib = "Date & time YYYY-MM-DD HH:MM"
        return DecodeCP32(value)
    elif dataType == 0x06: # INT48 = CP48.
        #vib = "Date & time YYYY-MM-DD HH:MM:SS, day, week, period"
        return DecodeCP48(value)
    elif dataType == 0x0E: # BCD12 = Unsupported date and time.
        #vib = "Date & time YY-MM-DD HH:MM:SS"
        return DecodeBCD12(value)

def loopVIFE(last, recordLength, arr, physicalQuantity, unit, exponent):
    vife = last
    while (vife & 0x80) > 0:
        vife = arr[recordLength]
        recordLength += 1
        if vife == 0xFF:
            vife = arr[recordLength]
            recordLength += 1
            physicalQuantity += ", manufacturer specific: %d" % (vife)
        else:
            if vife != 0xFC:
                # Combinable (orthogonal) VIFE-codes.
                section = "VIFE"
            else:
                # Extension Combinable (orthogonal) VIFE-codes (0xFC).
                vife = arr[recordLength]
                recordLength += 1
                section = "VIFE_FC"
            physicalQuantity, unit, exponent = getCombinableVIFE(vife, section, physicalQuantity, unit, exponent)  
    return recordLength, physicalQuantity, unit, exponent

# Decode object
def decode_field(arr,  opcname, telegramCounter):
    dif=arr[0]      # Get DIF.
    #print("\tDIF\t\t= 0x%02X"%dif)
    value=None
    exponentStr = ""
    humanvalue = None
    dataTypeStr=""              # String representation of the data type.
    recordLength=1              # Point to second byte in arr, may be DIFE or VIF.
    more_telegrams=False
    mfct_data=False
    functionFields=["Instantaneous value", "Maximum value", "Minimum value", "Value during error state"]
    function=functionFields[(dif>>4)&0x03]
    dataType = dif & 0x0F       # Get data type.
    storageNumber = (dif>>6)& 0x01
    subunit = 0
    tariff = 0
    vib = ""
    physicalQuantity = ""
    unit = ""
    isDateTime = False
    exponent = None
    abb_version_data = ""
    abb_version_object = False
    #hasManufacturerSpecificVIB = False

    try:
        # === Parse DIB ===========================================================
        if dataType == 0x0F: # DIF = MDH.
            value = "None"
            mdh = (dif>>4)&0x07  # Get MDH.
            dataTypeStr = "0x%sF"%(mdh)
            if mdh==0: # MDH = 0x0F.
                function="Manufacturer specific, last telegram"
                mfct_data=True
                recordLength = len(arr) - 2
                value = ""
                for i in range(recordLength):
                    value += "%02X "%struct.unpack("B", arr[i:i+1])[0]
                vib = "Manufacturer specific, last telegram"
            elif mdh==1: # MDH = 0x1F.
                function="Manufacturer specific, more telegrams"
                mfct_data=True
                more_telegrams=True
                recordLength = len(arr) - 2
                value = ""
                for i in range(recordLength):
                    value += "%02X "%struct.unpack("B", arr[i:i+1])[0]
                vib = "Manufacturer specific, more telegrams"
            elif mdh==2: # MDH = 0x2F.
                function="Idle filler, following byte is a DIF"
                vib = "Idle filler"
                value = None
            elif mdh==7: # MDH = 0x7F.
                function="Global readout request"
                vib = "Global readout request"
            else:
                function="Reserved"
                vib = "Reserved"

            return_map = {
                "value": value,
                "humanvalue" : value,
                "dataTypeStr": dataTypeStr,
                "recordLength": recordLength,
                "function": function,
                "mfct_data": mfct_data,
                "more_telegrams": more_telegrams,
                "vib": vib,
                "exponent": "",
                "templateVIB": "mfct",
                "abb_version_data": abb_version_data
            }
            return return_map

        elif dif & 0x80: # If DIFE is present, check all DIFE.
            difeCnt = 0
            for i in range(10):
                dife=arr[recordLength]              # Get DIFE.
                #print("\tDIFE\t\t= 0x%02X"%dife)
                recordLength += 1     # Point to next byte.
                storageNumber += (dife & 0x0f) << (1+difeCnt*4)
                tariff += ((dife & 0x30)>>4) << (difeCnt*2)
                subunit += ((dife & 0x40)>>6) << (difeCnt)
                difeCnt += 1
                # Check if extension bit is set in current DIFE.
                if (dife&0x80)==0:
                    break
        if(storageNumber != 0):
            function += ", Storage %d" % (storageNumber)
        if(tariff != 0):
            function += ", Tariff %d" % (tariff)
        if(subunit != 0):
            function += ", Subunit %d" % (subunit)
        #print(function)

        # === Parse VIB ===========================================================
        vif=arr[recordLength]  # Get VIB
        recordLength += 1
        last = vif
        
        if (vif == 0xFF and 
            arr[recordLength+1] == 0xAA and 
            arr[recordLength+2] == 0x00 and 
            dataType == 0x0D and
            telegramCounter == 1):
            
            abb_version_object = True

        templateVIB = ""
        if vif < 0xFB and vif != 0x7F and vif != 0x7C:  # Primary VIF-codes.
            physicalQuantity, unit, exponent,  isDateTime,  templateVIB = getVIF(vif)
        elif vif == 0xFB: # First extension of VIF-codes.
            vife = arr[recordLength]
            recordLength += 1
            last = vife
            physicalQuantity, unit, exponent,  isDateTime,  templateVIB = getVIFE_FB(vife)
        elif vif == 0x7C:   # Plain ASCII.
            # VIF LEN ASCII_1 ... ASCII_N DATA         (No VIFE present).
            length = arr[recordLength]
            recordLength += 1
            # Point first byte after length and include as many bytes as specified in length.
            ascii = arr[recordLength:recordLength+length]
            #vib, exponent = getPlainASCII(length, ascii)
            physicalQuantity = getPlainASCII(length, ascii)
            templateVIB = getPlainASCII_1(length, ascii)
            recordLength += length
        elif vif == 0xFC:
            # Note:
            # According to the EN13757-3 2018 section C2 it is very clear that the whole VIB (VIF + VIFEs)
            # should be intact, and the Plain text VIF should be placed after. However, some devices, including
            # Elvaco, places the Plain text VIF directly after the 0xFC VIF, but before the ortogonal VIFE.
            # Current decision is to support the Elvaco approach, instead of the standard.
            # See Issues #240, #442 and #464
            #
            # The VIFE will follow after the length and data.
            # Ex1: VIF LEN ASCII_1 ... ASCII_N VIFE DATA    (VIFE present).
            # Ex2: VIF LEN ASCII_1 ... ASCII_N DATA         (No VIFE present).
            length = arr[recordLength]
            # Point first byte after length and include as many bytes as specified in length.
            ascii = arr[recordLength+1:recordLength+1+length]
            #vib, exponent = getPlainASCII(length, ascii)
            physicalQuantity = getPlainASCII(length, ascii)
            templateVIB = getPlainASCII_1(length, ascii)
            recordLength += length + 1
        elif vif == 0xFD: # Second extension of VIF-codes.
            vife = arr[recordLength]
            recordLength += 1
            last = vife
            physicalQuantity, unit, exponent,  isDateTime,  templateVIB = getVIFE_FD(vife)
        elif vif == 0xEF: # (Reserved) Third extension VIF-codes.
            vife = arr[recordLength]
            recordLength += 1
            last = vife
            physicalQuantity, unit, exponent,  isDateTime,  templateVIB = getVIFE_EF(vife)
        elif vif == 0x7E or vif == 0xFE: # Any VIF.
            physicalQuantity = "Any VIF"
        elif vif == 0x7F:
            physicalQuantity, unit, exponent,  isDateTime,  templateVIB = getVIF(vif)
        elif vif == 0xFF: # Manufacturer specific.
            #vib = "Manufacturer specific"
            vife = arr[recordLength]
            recordLength += 1
            last = vife
            physicalQuantity, unit, exponent,  isDateTime,  templateVIB = getVIFE_FF(vife, opcname)
            #hasManufacturerSpecificVIB = True
        else: # Unknown VIF.
            physicalQuantity = "Unknown"

        recordLength, physicalQuantity, unit, exponent = loopVIFE(last, recordLength, arr, physicalQuantity, unit, exponent)

        if unit == "":
            vib = physicalQuantity
        else:
            vib = physicalQuantity + " [" + unit + "]"
        
        if exponent != None:
            exponentStr = "E%d" % (exponent)

        # === Parse DATA ==================================================
        # No data.
        #printhex(arr)
        if dataType==0:
            dataTypeStr="No Data"
        elif dataType==0x01: # INT8.
            dataTypeStr="INT8"
            value=struct.unpack('<b',arr[recordLength:recordLength+1])
            value = value[0]
            humanvalue = human_value(value, exponent)
            recordLength += 1
        elif dataType==0x02: # INT16.
            #printhex(arr[recordLength:])
            dataTypeStr="INT16"
            value=struct.unpack('<h',arr[recordLength:recordLength+2])
            value = value[0]
            humanvalue = human_value(value, exponent)
            recordLength +=  2
        elif dataType==0x03: # INT24.
            dataTypeStr="INT24"
            # unpack as 32-bit. Pad with zero before to avoid sign extension. Then shift after unpack
            value = struct.unpack('<i', b'\x00' + arr[recordLength:recordLength + 3])
            value = value[0] >> 8
            humanvalue = human_value(value, exponent)
            recordLength += 3
        elif dataType==0x04: # INT32.
            #printhex(arr)
            #printhex(arr[recordLength:])
            dataTypeStr="INT32"
            value=struct.unpack('<i',arr[recordLength:recordLength+4])
            value = value[0]
            humanvalue = human_value(value, exponent)
            recordLength += 4
        elif dataType==0x05: # FLOAT.
            dataTypeStr="REAL32"
            value=struct.unpack('<f',arr[recordLength:recordLength+4])
            value = value[0]
            humanvalue = value * pow(10,exponent)
            recordLength += 4
        elif dataType==0x06: # INT48.
            #print("HHHH INT48")
            #printhex(arr[recordLength:recordLength+6])
            dataTypeStr="INT48"
            # unpack as 64-bit. Pad with zero before to avoid sign extension. Then shift after unpack
            value=struct.unpack('<q',b'\x00\x00' + arr[recordLength:recordLength+6])
            value = value[0] >> 16
            humanvalue = human_value(value, exponent)
            #print("IIII")
            recordLength += 6
        elif dataType==0x07:  # INT64.
            dataTypeStr="INT64"
            value=struct.unpack('<q',arr[recordLength:recordLength+8])
            value = value[0]
            humanvalue = human_value(value, exponent)
            recordLength += 8
        elif dataType==0x08: # Selection for readout.
            #dataTypeStr="Selection for Readout"
            dataTypeStr="SEL"
            recordLength = recordLength + 8
        elif dataType==0x09:  # BCD2.
            dataTypeStr="BCD2"
            value=bcdtoval(arr[recordLength:recordLength+1])
            humanvalue = human_value(value, exponent)
            recordLength += 1
        elif dataType==0x0A: # BCD4.
            dataTypeStr="BCD4"
            value=bcdtoval(arr[recordLength:recordLength+2])
            humanvalue = human_value(value, exponent)
            recordLength += 2
        elif dataType==0x0B: # BCD6.
            dataTypeStr="BCD6"
            value=bcdtoval(arr[recordLength:recordLength+3])
            humanvalue = human_value(value, exponent)
            recordLength = recordLength + 3
        elif dataType==0x0C: # BCD8.
            dataTypeStr="BCD8"
            value=bcdtoval(arr[recordLength:recordLength+4])
            humanvalue = human_value(value, exponent)
            recordLength += 4
        elif dataType==0x0D:  # Variable length.
            LVAR=arr[recordLength]
            dataLength = 0
            recordLength += 1
            dataTypeStr = "LVAR"
            # There is a bug in Browse. The LVAR container for wireless is not shown correctly.
            if arr[recordLength-3] == 0xFD and arr[recordLength-2] == 0x3B: # Wireless container.
                dataLength = LVAR
                data = arr[recordLength:recordLength+dataLength]
                value = "".join("%02X "%(i) for i in data)            
            elif LVAR <= 0xBF:  # 8-bit text (ISO 8859-1).
                dataLength = LVAR
                #print("dataLength in 0x0D", dataLength)
                data = arr[recordLength:recordLength+dataLength]
                nonPrintable = False
                for c in data:
                    if c<0x20 or c>0x7E or c==0x22 or c==0x27 or c==0x5C:
                        #Only print ASCII characters between " " and "}", but avoid """, "'" and "\"
                        nonPrintable = True
                # Save the ABB version
                if abb_version_object:
                    #print("abb_version_object True in 0x0D")
                    #print("data", data)
                    abb_version_data = data
                value = "".join("%02X "%(i) for i in data[::-1])   # Byte ouput.
                if (nonPrintable):
                    humanvalue = value
                else:
                    humanvalue = "".join(chr(i) for i in data[::-1])         # Char ouput.
            elif LVAR >= 0xC0 and LVAR <= 0xCF: # Positive BCD number.
                dataLength = LVAR & 0x0F
                data = arr[recordLength:recordLength+dataLength]
                value = "".join("%02X"%(i) for i in data[::-1])
                humanvalue = value
            elif LVAR >= 0xD0 and LVAR <= 0xDF: # Negative BCD number.
                dataLength = LVAR & 0x0F
                data = arr[recordLength:recordLength+dataLength]
                value = "-" + "".join("%02X"%(i) for i in data[::-1])
                humanvalue = value
            elif LVAR >= 0xE0 and LVAR <= 0xEF:  # Binary number with (LVAR - 0xE0) bytes.
                dataLength = LVAR & 0x0F
                data = arr[recordLength:recordLength+dataLength]
                value = "".join("%02X "%(i) for i in data[::-1])   # Byte ouput.
                humanvalue = value
            elif LVAR >= 0xF0 and LVAR <= 0xF4: # Binary number with 4*(LVAR - 0xEC) bytes.
                dataLength = (LVAR - 0xEC) * 4
                data = arr[recordLength:recordLength+dataLength]
                value = "".join("%02X "%(i) for i in data[::-1])   # Byte ouput.
                humanvalue = value
            elif LVAR == 0xF5: # Binary number with 48 bytes.
                dataLength = 48
                data = arr[recordLength:recordLength+dataLength]
                value = "".join("%02X "%(i) for i in data[::-1])   # Byte ouput.
                humanvalue = value
            elif LVAR == 0xF6: # Binary number with 64 bytes.
                dataLength = 64
                data = arr[recordLength:recordLength+dataLength]
                value = "".join("%02X "%(i) for i in data[::-1])   # Byte ouput.
                humanvalue = value
            else: # Reserved
                value = "Reserved"
                humanvalue = value
            recordLength += dataLength
        elif dataType==0x0E: # BCD12.
            dataTypeStr="BCD12"
            value=bcdtoval(arr[recordLength:recordLength+6])
            humanvalue = human_value(value, exponent)
            recordLength += 6

        # Decode value to date or time.
        if isDateTime and (dataType == 0x02 or dataType == 0x03 or dataType == 0x04 or dataType == 0x06 or dataType == 0x0E):
            value = DecodeDateTime(dataType, value)
            humanvalue = value

        # Add unit also to human radable value field
        if unit != "":
            humanvalue = str(humanvalue) + " " + unit

    except:
        function = "Decoding error"
        value = "".join("%02X "%(i) for i in arr)   # Byte ouput.
        humanvalue = value
        dataTypeStr = ""
        vib = ""
        templateVIB=""
        exponent = ""
        recordlen = len(arr)

        print("Exception")

    return_map = {
        "value": value,
        "humanvalue" : humanvalue,
        "dataTypeStr": dataTypeStr,
        "recordLength": recordLength,
        "function": function,
        "mfct_data": mfct_data,
        "more_telegrams": more_telegrams,
        "vib": vib,
        "exponent": exponentStr,
        "templateVIB": templateVIB,
        "abb_version_data": abb_version_data
    }
    return return_map
