from shutil import copyfile
from uuid import getnode
import configparser
import os, re, time, glob
from werkzeug.utils import secure_filename
from flask import Flask, render_template, request, send_file, redirect, url_for, render_template_string, after_this_request, json, make_response,jsonify
from flask import Response
from flask import session
import weblib
import sys

from password_manager import PasswordManager
import codecs
import subprocess
import traceback
import importlib
import shutil
import re
from ipaddress import ip_address
import hashlib
import datetime
import api.system_api
import lib.piigab_900_system as piigab_system

import logging
import logging.handlers
logging.basicConfig(
    handlers=[
        logging.handlers.RotatingFileHandler(
            filename= "/var/log/webinterface.log",
            mode="a",
            maxBytes=1024*20, # kb max
            backupCount=3,
            delay=False
            )
    ],
    format='%(asctime)s %(name)s %(levelname)-8s %(message)s',
    level=logging.DEBUG,
    datefmt='%Y-%m-%d %H:%M:%S')


def set_date_and_time(date,t):
    """sets date & time and updates hw clock acordingly

        date (str): '%Y:%m:%d'

        t (str): '%H:%M:%S'
    """
    # for validation, throws error if invalid.
    valid_date = datetime.datetime.strptime(date + " " +t, '%Y-%m-%d %H:%M:%S')
    if valid_date.year <= 2011:
        # safeguard against bad date. 
        # see https://github.com/pallets/itsdangerous/issues/102
        raise Exception("Year is invalid")
    
    print("valid date:" , valid_date)
    code = subprocess.call(["date","--set=%s"%str(valid_date) ])

    print("setting date" , code)
    code = subprocess.call(["hwclock","-u","-w","--rtc","/dev/rtc0"])
    print("setting rtc0" , code)
    code = subprocess.call(["hwclock","-u","-w","--rt","/dev/rtc1"])
    print("setting rtc1" , code)

""" holds data about the systems selected zone. """
ZONEINFO_FILE = weblib.PROOT + "/etc/zoneinfo.json"
# if year is less or equal to 2011 it will crash with itsdangerous v0.24
if datetime.datetime.now().year <= 2011:
    print("Year was less or equal to 2011, adjusting date...")
    set_date_and_time("2023-09-25","00:00:00")

app = Flask(__name__, static_url_path='')

from flask_compress import Compress
Compress(app)

app.logger.info("Starting server")

app.jinja_env.add_extension('jinja2.ext.loopcontrols')
app.jinja_env.add_extension('jinja2.ext.do')

app.jinja_env.lstrip_blocks = True
app.jinja_env.trim_blocks = True
app.config["SESSION_COOKIE_NAME"] = 'auth_token'
app.config["SESSION_COOKIE_HTTPONLY"] = True
app.config["SESSION_COOKIE_SECURE"] = True
app.config["SESSION_COOKIE_SAMESITE"] = "Strict"
app.secret_key = os.urandom(64)



class SessionManager:
    count = 0
    active = set()

    def new(self):
        count = self.count
        self.active.add(count)
        self.count += 1
        return count

class Software:
    def __init__(self, name ,pretty_name, is_installed, link_url, endpoint):
        self.name = name
        self.is_installed = is_installed
        # endpoint is the blueprint_name.the_method_route_name
        # used for menu.html macro to determine the current active page. for menu
        self.endpoint = endpoint
        self.pretty_name = pretty_name
        self.link_url = link_url


installed_programs = [
    Software(name="mbushub",pretty_name="MBusHub",is_installed=False,link_url="/mbushub_portconf?SLNR=0", endpoint=""),
    Software(name="wireless",pretty_name="Wireless",is_installed=False,link_url="/wireless_config", endpoint=""),
    Software(name="quickpost",pretty_name="QuickPost",is_installed=False,link_url="/quickpost_config", endpoint=""),
    Software(name="mbusascii2mqtt",pretty_name="MBus2MQTT",is_installed=False,link_url="/mbusascii2mqtt_config", endpoint=""),
    Software(name="modbus2mbus",pretty_name="Modbus2MBus",is_installed=False,link_url="/modbus2mbus_config", endpoint=""),
    Software(name="alarmserver",pretty_name="AlarmServer",is_installed=False,link_url="/alarmserver_config", endpoint=""),
    Software(name="dlms2mbus",pretty_name="DLMS2MBus",is_installed=False,link_url="/dlms2mbus_config", endpoint=""),
    Software(name="lte",pretty_name="GSM/LTE", is_installed=False,link_url="/lte_config", endpoint=""),
    Software(name="lora",pretty_name="LoRa", is_installed=False,link_url="/lora_config", endpoint=""),
    Software(name="han2mbus",pretty_name="Han2MBus", is_installed=False,link_url="/han2mbus_config", endpoint=""),
    Software(name="microservicebus", pretty_name="Portal Client", is_installed=False, link_url="/msb_config", endpoint=""),
    Software(name="mbus2bacnet", pretty_name="MBus2Bacnet", is_installed=False, link_url="/mbus2bacnet_config", endpoint=""),

]


for software in installed_programs:
    name = software.name
    if os.path.exists("%s/%s.py"%(name,name)):
        try:
            #print(name)
            a=importlib.import_module("%s.%s"%(name,name))#, package='app')
            app.register_blueprint(getattr(a,"%s_blueprint"%name))
            software.is_installed = True
            software.endpoint = name + "."
        except Exception as e:
            traceback.print_exc()

app.register_blueprint(api.system_api.blueprint)


@app.errorhandler(500)
def server_error(error):
    app.logger.exception('An exception occurred during a request.')
    log = ""
    try:
        with open('/var/log/webinterface.log') as f:
            log = f.read()
    except Exception:
        log = "Failed to read log"
    
    return render_template('internal_server_error.html', 
                           log=log,
                           serial=weblib.serialNo(),
                           pi900_version=weblib.pi900_version,
                           pi900_type=weblib.pi900_type,
                           pi900_os=weblib.pi900_os,
                           pi900_os_name=weblib.PI900_OS_NAME
                           ), 500

def get_installed_software():
    return list(filter(lambda software: software.is_installed, installed_programs))


#@app.context_processor
def get_identification():
    try:
        with open("/var/www1/identification.py", "r") as file:
            return file.read()
    except Exception:
        return ""

def get_pi900_type():
    return weblib.pi900_type

app.jinja_env.globals["identification"] = get_identification
app.jinja_env.globals["installed_software"] = get_installed_software
app.jinja_env.globals["get_pi900_type"] = get_pi900_type
app.jinja_env.globals["is_new"] = True
app.jinja_env.globals['serial_number'] = weblib.serialNo()


try:
    from core.search import search_blueprint
    app.register_blueprint(search_blueprint)
except Exception as e:
    print("failed to import search blueprint in main. ", str(e))

password_manager = PasswordManager(PROOT=weblib.PROOT)
session_manager = SessionManager()

@app.before_request
def require_basic_auth():
    if request.host in [ 
            #piigab_system.get_vpn_interface_address('/usr/local/etc/msb-py/'), 
            '127.0.0.1', 'localhost'
        ]:
        return
    
    auth = request.authorization
    aid = session.get('auth')

    if aid and aid in session_manager.active:
        return

    elif not auth:
        return Response(status=401, headers={'WWW-Authenticate': 'Basic realm="pi-900"'})

    elif password_manager.is_rate_limited():
        return Response(status=429, headers={'Retry-After': str(password_manager.timeout)})

    else:
        if password_manager.check(auth.username, auth.password):
            session['auth'] = session_manager.new()
            session.modified = True
            password_manager.success()

        else:
            if session.get('auth'): session.pop('auth', None)
            password_manager.fail()
            return Response(status=401, headers={'WWW-Authenticate': 'Basic realm="pi-900"'})

def allowed_file(filename):
    return '.' in filename and \
           filename.rsplit('.', 1)[1].lower() in ['csv','ini','xml']

@app.route("/incorrect_installation",  methods=['GET'])
def incorrectInstallation():
    software = request.args.get("software")
    return render_template("problems_install_software.html",  software=software)


@app.route("/")
def startPage():

    #Administration card

    serialno=0
    clients=0
    meters=0
    protocols=0
    try:
        mbs=os.popen("/binary/mbushub.elf -S")
        serialno=mbs.readline().strip('\n')
        mbs.close()
        mbl=os.popen("/binary/mbushub.elf -l")
        licensearray=mbl.readline().strip('\n').split(';')
        mbl.close()
        clients=licensearray[0]
        meters=licensearray[1]
        protocols=licensearray[2].split(',')
        length = len(licensearray)
        for i in range(length):
            print(licensearray[i])
    except:
        pass

    #Meterlist card
    numberOfMeters = 0
    numberOfWiredMeters = 0
    numberOfRadioMeters = 0
    numberOfNodes = 0
    numberOfUndefinedMeters = 0

    meterlist = []
    try:
        with open("/config/meterlist.csv") as file:
            delimiter = weblib.sniffer_csv(file)
            meterlist = list(weblib.rcf(file, delimiter))
        meterlist = sorted(meterlist)
    except Exception as e:
        print("error in start_page "+str(e))

    wiredMeters=[]
    radioMeters=[]
    nodes=[]
    undefinedMeters=[]
    for row in meterlist:
        if len(row) > 8:
            if row[8] == "W":
                wiredMeters.append(row)
                numberOfWiredMeters += 1
                numberOfMeters +=1
            elif row[8] == "R":
                radioMeters.append(row)
                numberOfRadioMeters += 1
                numberOfMeters +=1
            elif row[8] == "N":
                nodes.append(row)
                numberOfNodes += 1
                numberOfMeters +=1
            else:
                undefinedMeters.append(row)
                numberOfUndefinedMeters +=1
                numberOfMeters +=1
        else:
            undefinedMeters.append(row)
            numberOfUndefinedMeters +=1
            numberOfMeters +=1

   #Port configuration card

    try:
        mbv=os.popen("/binary/mbushub.elf -V")
        mbusHubVersion=mbv.read()
        mbv.close()
        mbv=os.popen("/binary/mbushub.elf -a")
        licarray=mbv.read().split(',')
        mbv.close()
        nslaves = int(licarray[1])
    except:
        nslaves = 0
        mbusHubVersion = ''
        licarray = ''

    mbushubini_def = configparser.RawConfigParser()
    mbushubini_def.read("/config/mbushub_default.ini")
    mbushubini = configparser.RawConfigParser()
    mbushubini.read("/config/mbushub.ini")

    slaveport = {
        "porttype": str,
        "protocol": str,
        "timeout": int,
        "localport": int,
        "sercomport": str,
        "serbaudrate": int
    }

    try:
        parser = weblib.Configuration(mbushubini_def, mbushubini)
        config = parser.read({
            "MasterPort": {
                "porttype": str,
                "protocol": str,
                "timeout": int,
                "sercomport": str,
                "serbaudrate": int,
                "remoteip": str,
                "remoteport": int
            },
            "SlavePort1": slaveport,
            "SlavePort2": slaveport
        })
    except:
        traceback.print_exc()

    if nslaves > 2:
        tmp = parser.read({ "SlavePort3": slaveport })
        config["SlavePort3"] = tmp["SlavePort3"]
        if nslaves > 3:
            tmp = parser.read({ "SlavePort4": slaveport })
            config["SlavePort4"] = tmp["SlavePort4"]

    try:
        quickpostini = configparser.RawConfigParser()
        quickpostini.read("/config/quickpost.ini")
        quickpostPort = quickpostini["Main"]["mbus_ascii_server_port"]
    except:
        quickpostPort = ""
###
    di1=weblib.getGPIO("pioA3")
    di2=weblib.getGPIO("pioA4")
    relay  = "Undefined"
    try:
        rnc=weblib.getGPIO("pioC12")
        rno=weblib.getGPIO("pioC10")
        if  (rno == 'On' and rnc == 'Off'):
            relay  = "On"
        elif (rno == 'Off' and rnc == 'On'):
            relay = "Off"
    except:
        pass

    # M-Bus Loop Power
    rs485_failsafe = weblib.getGPIO("pioA24")
    loop1 = weblib.getGPIO("pioA26")
    loop2 = weblib.getGPIO("pioA27")
    loop3 = weblib.getGPIO("pioA28")
    loop4 = weblib.getGPIO("pioA29")
    # End of M-Bus Loop Power

    board_temperature = "Missing"
    mbus_stage_temperature = "Missing"
    error_flag = 0
    try:
        conf = configparser.RawConfigParser()
        conf.read('/tmp/pi900_state.txt')
        board_temperature = conf.get('Main','Temperature_40V')
        mbus_stage_temperature = conf.get('Main','Temperature_MBus')
        error_flag  = int(conf.get('Main','MBusErrorVal'))
    except:
        pass

    error_message="0, No error"
    if (error_flag & 1):
        error_message = "M-Bus Overload"
    if (error_flag & 2):
        error_message = "M-Bus Short Circuit"
    error_string = "%08X"%error_flag
    mbusCurrent = 0
    mbusMaxCurrent = 0
    try:
        mbusCurrent      = float(conf.get('Main','MbusCurrent'))
        mbusMaxCurrent   = float(conf.get('Main','MbusMaxCurrent'))
    except:
        pass

    mbusCurrentString      = "%.1f mA"%mbusCurrent
    mbusMaxCurrentString   = "%.1f mA"%mbusMaxCurrent
    mbusCurrentPercentUsed = "Unavailable"
    if mbusMaxCurrent > 0:
        mbusCurrentPercentUsed = "%d%%"%int(mbusCurrent/mbusMaxCurrent * 100)

###
    info = {
        'di1'    : di1,
        'di2'    : di2,
        'relay'  : relay,
        'loop1'  : loop1,
        'loop2'  : loop2,
        'loop3'  : loop3,
        'loop4'  : loop4,
        'rs485_failsafe' : rs485_failsafe,
        'board_temperature' : board_temperature,
        'mbus_stage_temperature' : mbus_stage_temperature,
        'error_flag' : error_flag,
        'error_message' : error_message,
        'error_string' : error_string,
        'mbusCurrent' : mbusCurrent,
        'mbusMaxCurrent' : mbusMaxCurrent,
        'mbusCurrentString'      : mbusCurrentString,
        'mbusMaxCurrentString'   : mbusMaxCurrentString,
        'mbusCurrentPercentUsed' : mbusCurrentPercentUsed,
        'mbusHubVersion' : mbusHubVersion,
        'nslaves' : nslaves,
        'numberOfMeters' : numberOfMeters,
        'numberOfWiredMeters' : numberOfWiredMeters,
        'numberOfRadioMeters' : numberOfRadioMeters,
        'numberOfNodes' : numberOfNodes,
        'numberOfUndefinedMeters' : numberOfUndefinedMeters,
        'quickpostPort' : quickpostPort,
        'serialno': serialno,
        'clients': clients,
        'meters': meters,
        'protocols': protocols,
        'DeviceTypes' : weblib.DeviceTypes,
        'csvDict' : weblib.csvDict,
        'wiredMeters' : wiredMeters,
        'nodes' : nodes,
        'radioMeters' : radioMeters,
        'undefinedMeters' : undefinedMeters,
        'porttypeM' : config["MasterPort"]["porttype"],
        'protocolM' : config["MasterPort"]["protocol"],
        'timeoutM' : config["MasterPort"]["timeout"],
        'sercomportM' : config["MasterPort"]["sercomport"],
        'serbaudrateM' : config["MasterPort"]["serbaudrate"],
        'remoteipM' : config["MasterPort"]["remoteip"],
        'remoteportM' : config["MasterPort"]["remoteport"],
        'porttypeS1' : config["SlavePort1"]["porttype"],
        'protocolS1' : config["SlavePort1"]["protocol"],
        'timeoutS1' : config["SlavePort1"]["timeout"],
        'localportS1' : config["SlavePort1"]["localport"],
        'sercomportS1' : config["SlavePort1"]["sercomport"],
        'serbaudrateS1' : config["SlavePort1"]["serbaudrate"],
        'porttypeS2' : config["SlavePort2"]["porttype"],
        'protocolS2' : config["SlavePort2"]["protocol"],
        'timeoutS2' : config["SlavePort2"]["timeout"],
        'localportS2' : config["SlavePort2"]["localport"],
        'sercomportS2' : config["SlavePort2"]["sercomport"],
        'serbaudrateS2' : config["SlavePort2"]["serbaudrate"],
        'pi900_version' : weblib.pi900_version,
        'pi900_type' : weblib.pi900_type,
        'pi900_os' : weblib.pi900_os,
        'PI900_OS_NAME' : weblib.PI900_OS_NAME
    }
    if nslaves > 2:
        port3Info = {
            'porttypeS3' : config["SlavePort3"]["porttype"],
            'protocolS3' : config["SlavePort3"]["protocol"],
            'timeoutS3' : config["SlavePort3"]["timeout"],
            'localportS3' : config["SlavePort3"]["localport"],
            'sercomportS3' : config["SlavePort3"]["sercomport"],
            'serbaudrateS3' : config["SlavePort3"]["serbaudrate"]
        }
        info.update(port3Info)
    if nslaves > 3:
        port4Info = {
            'porttypeS4' : config["SlavePort4"]["porttype"],
            'protocolS4' : config["SlavePort4"]["protocol"],
            'timeoutS4' : config["SlavePort4"]["timeout"],
            'localportS4' : config["SlavePort4"]["localport"],
            'sercomportS4' : config["SlavePort4"]["sercomport"],
            'serbaudrateS4' : config["SlavePort4"]["serbaudrate"]
        }
        info.update(port4Info)
    return render_template('start_page.html', **info )


@app.route("/administration", methods=['GET'])
def administration():

    #CONFIGURATION
    update = request.args.get("UPDATE")
    tab = request.args.get("TAB")

    csvArray=glob.glob("/config/*.[cC][sS][vV]")+glob.glob("/config/*.[xX][mM][lL]")
    for i in range(len(csvArray)):
        csvArray[i] = os.path.basename(csvArray[i])
    iniArray = glob.glob("/config/*.[iI][nN][iI]")
    for i in range(len(iniArray)):
        iniArray[i] = os.path.basename(iniArray[i])

    tzstr=""
    hwtime=""
    ntpIP=""
    try:
        f=open("/%s/etc/TZ"%weblib.PROOT,'r')
        tzstr=f.read().strip()
        f.close()
    except:
        pass
    try:
        #tzstr = "UTC+1"
        cf=os.popen("/sbin/hwclock --rtc /dev/rtc1 2>/dev/null")
        hwtime=cf.read().rstrip()
        cf.close()
    except:
        pass
    try:
        f=open("/%s/etc/ntp.conf"%weblib.PROOT,'r')
        a=f.read()
        f.close()
        ntpIP=a.split('\n')[0].split()[1]
    except:
        pass

    #UPDATE PAGE
    if update == "TRUE":
        return render_template_string(
            """<HTML>
                    <HEAD>
                        <meta http-equiv=\"refresh\" content=\"0;url=/administration\">"
                    </HEAD>
                </HTML>""")
    inifileArray=glob.glob("/config/*.[iI][nN][iI]")
    for i in range(len(inifileArray)):
        inifileArray[i] = os.path.basename(inifileArray[i])

    securityEnabled = False
    passwordEmpty = True #False if password has been set
    currtime = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")

    mbushubVersion=""
    quickpostVersion=""
    modbusVersion=""
    wirelessVersion=""
    hanbusVersion=""

    try:
        mbv=os.popen("/binary/mbushub.elf -V")
        mbushubVersion=mbv.read()
        mbv.close()
    except:
        pass
    try:
        qv=os.popen("/binary/quickpost.elf -v")
        quickpostVersion=qv.read()
        qv.close()
    except:
        pass
    try:
        m2m=os.popen("/binary/modbus2mbus -v")
        modbusVersion=m2m.read()
        m2m.close()
    except:
        pass
    try:
        wv=os.popen("/binary/wireless -v")
        wirelessVersion=wv.read()
        wv.close()
    except:
        pass
    try:
        h2m=os.popen("/binary/han2mbus -v")
        hanbusVersion=h2m.read()
        wv.close()
    except:
        pass

    pingIntervalArray = ['Unset','30m','2h','6h','24h']
    ping_address = "Missing" # Used if config file is missing
    ping_interval = "Unset"
    eth_info = piigab_system.get_interfaces_info()
    try:
        cnf = weblib.Configs("config","pi900_system")
        ping_address = cnf.getopt('PingReboot','ping_address')
        ping_interval = cnf.getopt('PingReboot','ping_interval')
        if ping_interval not in pingIntervalArray: # This should never happen
            ping_interval='Unset'
        if weblib.validIPAddress(ping_address) != "IPv4":
            ping_address = eth_info['eth0GW']
    except:
        pass

    
    info = {
        'pingIntervalArray' : pingIntervalArray,
        'ping_address': ping_address,
        'ping_interval': ping_interval,
        'mbushubVersion': mbushubVersion,
        'quickpostVersion': quickpostVersion,
        'modbusVersion': modbusVersion,
        'wirelessVersion': wirelessVersion,
        'hanbusVersion': hanbusVersion,
        'update': update,
        'tab': tab,
        'randip' : piigab_system.is_randomized_ip("/%s/etc/randip"%weblib.PROOT),
        'csvArray' : csvArray,
        'iniArray' : iniArray,
        'tzstr': tzstr,
        'hwtime': hwtime,
        'ntpIP': ntpIP,
        'inifileArray': inifileArray,
        'securityEnabled': securityEnabled,
        'passwordEmpty': passwordEmpty,
        'currtime': currtime,
        'pi900_type' : weblib.pi900_type,
        'pi900_os' : weblib.pi900_os,
        'pi900_ipk' : weblib.pi900_ipk,
        'is_default_password' : password_manager.has_default_password('Admin')
    }
    info.update(eth_info)
    return render_template('administration.html', **info)


@app.route("/licenses", methods=['GET'])
def licenses_route():
    try:
        name = secure_filename(os.path.basename(request.args.get('FILE')))
        return send_file("/var/www1/licenses/" + name)
    except:
        info = { 'license_files': [] }
        for entry in os.listdir("/var/www1/licenses/"):
            info['license_files'].append(entry)
        return render_template('licenses.html', **info)


@app.route("/showFile", methods=['GET'])
def showfile():
    cstr=""
    try:
        fname="/config/" + secure_filename(os.path.basename(request.args.get('FILE')))
        f=open(fname,'r')
        cstr=f.read()
        f.close()
    except:
        cstr="No Filename given"

    cstr=cstr.replace('\n','<br>')
    hstr="<HTML>\n<HEAD>\n</HEAD>\n<body>%s</body>\n</HTML>\n"%cstr
    return render_template_string(hstr)

@app.route("/basic_settings", methods=['GET'])
def base_settings():
    hwconfig = configparser.RawConfigParser()
    hwconfig.read('/config/mbushub_lic.ini')

    serialno=0
    clients=0
    meters=0
    protocols=0
    cards=0
    licensestring=""

    try:
        mbs=os.popen("/binary/mbushub.elf -S")
        serialno=mbs.readline().strip('\n')
        mbs.close()
        mbl=os.popen("/binary/mbushub.elf -l")
        licensearray=mbl.readline().strip('\n').split(';')
        mbl.close()
        clients=licensearray[0]
        meters=licensearray[1]
        protocols=licensearray[2]
        cards=licensearray[3]
        licensestring=hwconfig.get('Main', 'licstr')
    except:
        pass

    https_on=""
    http_on=""
    conffile = os.path.basename(os.readlink("/etc/lighttpd/lighttpd.conf"))
    print("webserver setting:", conffile)
    if (conffile == 'lighttpd_https.conf'):
        https_on="checked"
    elif (conffile == 'lighttpd_http.conf'):
        http_on="checked"
    form = {
        'serialno': serialno,
        'clients': clients,
        'meters': meters,
        'protocols': protocols,
        'cards': cards,
        'licensestring': licensestring,
        'https_on': https_on,
        'http_on': http_on,
        'conffile': conffile,
        'is_default_password' : password_manager.has_default_password('Admin')
    }
    return render_template('basic_settings.html', **form)

def line_prepender(filename, line):
    with open(filename, 'r+') as f:
        content = f.read()
        f.seek(0, 0)
        f.write(line.rstrip('\r\n') + '\n' + content)

@app.route("/downloadCsvFile")
def downloadCsvFile():
    cstr = secure_filename(os.path.basename(request.args.get('FILE')))
    cf = []
    temporary = ""
    if cstr == "meterlist.csv":
        try:
            header = "#Id-number,Manufacturer,Version,Medium,Manufacturer(text),Fabrication-number,Primary-address,Verified,Meter-type(Wired/Radio/Wireless Node),Wireless-key,Description,Timestamp"
            line_prepender("/config/meterlist.csv", header)
        except:
            pass

        try:
            return send_file('/config/%s'%cstr, attachment_filename=cstr, mimetype='application/octet-stream', as_attachment=True,  cache_timeout=0)
        except:
            return 'File not found'
    elif cstr == "myconfig.csv" or cstr == "masterpost_Mbus2Modbus.csv" or cstr == "slaveport_Mbus2Modbus.csv":
        with open("/config/"+cstr) as file:
            temporary = file.read()
            with open("/tmp/"+cstr+"_download", "w", encoding="latin-1") as file_write:
                file_write.write(temporary)

        try:
            return send_file('/tmp/%s_download'%cstr, attachment_filename=cstr, mimetype='application/octet-stream', as_attachment=True,  cache_timeout=0)
        except:
            return 'File not found'
    else:
        try:
            return send_file('/config/%s'%cstr, attachment_filename=cstr, mimetype='application/octet-stream', as_attachment=True,  cache_timeout=0)
        except:
            return 'File not found'

@app.route("/copyFromConfig")
def copyFromConfig():
    try:
        shutil.copyfile("/config/meterlist.csv","/tmp/meterlist.csv")
    except:
        pass
    return redirect('/meter_search_and_overview')

@app.route("/rmCsvFile")
def rmCsvFile():
    try:
        filename = secure_filename(os.path.basename(request.args.get('CSVRMF')))# "CSVRMF").replace(' ','\ ')
        os.remove("/config/"+filename)
    except Exception as e:
        print("exception", str(e))
    return render_template_string("""<HTML><HEAD>
            File removed successfully.
            <meta http-equiv=\"refresh\" content=\"2;url=/administration\">
            </HEAD></HTML>""")

@app.route("/backupConf", methods=['POST'])
def backupConf():
    mac = getnode()
    serialno = (mac >> 1) & 0x3FFFFFF
    fname = "pi%sconfig_%d_backup"%(weblib.pi900_type,serialno)
    subprocess.call(["tar","cf","/tmp/%s.tar"%fname,"%s/config"%weblib.PROOT])
    subprocess.call(["gzip","/tmp/%s.tar"%fname])
    shutil.move("/tmp/%s.tar.gz"%fname,"/tmp/%s.tgz"%fname)
    return send_file('/tmp/%s.tgz'%fname, attachment_filename='%s.tgz'%fname, mimetype='application/octet-stream', as_attachment=True)

@app.route("/changePassword", methods=['POST'])
def changePassword():
    try:
        password = request.json["pass1"]
        new = request.json["pass2"]
        check = request.json["pass3"]
    except Exception:
        return jsonify({"message": "Invalid parameters"}), 400

    # min length is 8
    if len(new) < 8:
        return jsonify({"message": "Password length must be atleast 8 charathers long."}), 400

    if new != check:
        return jsonify({"message": "passwords do not match"}), 400

    elif password_manager.check('Admin', password):
        password_manager.update('Admin', new)
        token = session.get('auth')
        if token:
            session_manager.active = set()
            session.pop('auth', None)
        return jsonify({"message": "Password updated."}), 200

    return jsonify({"message": "Old password incorrect."}), 400


@app.route("/defaultSettings")
def defaultSettings():
    result = """<HTML><HEAD>
            <meta http-equiv=\"refresh\" content=\"2;url=/basic_settings\">
        </HEAD>
        <BODY>"""
    val=""
    if request.args.get("DEFSET"):
        val=request.args.get("DEFSET")

    if val == 'system':
        result += "Setting system configuration to Default"
        shutil.copyfile("/%s/default/ntp.conf"%weblib.PROOT,"/%s/etc/ntp.conf"%weblib.PROOT)
        shutil.copyfile("/%s/default/TZ"%weblib.PROOT,"/%s/etc/TZ"%weblib.PROOT)
        shutil.copyfile("/%s/default/zoneinfo.json"%weblib.PROOT,"/%s/etc/zoneinfo.json"%weblib.PROOT)
    elif val == 'mbushub':
        result += "Setting MBusHub configuration to Default"
        if os.path.isfile("/%s/default/mbushub_default.ini"%weblib.PROOT):
            shutil.copyfile("/%s/default/mbushub_default.ini"%weblib.PROOT,"/config/mbushub_default.ini")
        shutil.copyfile("/config/mbushub_default.ini","/config/mbushub.ini")
    elif val == 'quickpost':
        result += "Setting Quickpost configuration to Default"
        if os.path.isfile("/%s/default/quickpost_default.ini"%weblib.PROOT):
            shutil.copyfile("/%s/default/quickpost_default.ini"%weblib.PROOT,"/config/quickpost_default.ini")
        shutil.copyfile("/config/quickpost_default.ini","/config/quickpost.ini")
    elif val == 'modbus2mbus':
        result += "Setting Modbus2MBus configuration to Default"
        if os.path.isfile("/%s/default/modbus2mbus_default.ini"%weblib.PROOT):
            shutil.copyfile("/%s/default/modbus2mbus_default.ini"%weblib.PROOT,"/config/modbus2mbus_default.ini")
        shutil.copyfile("/config/modbus2mbus_default.ini","/config/modbus2mbus.ini")
    elif val == 'wireless':
        result += "Setting Wireless configuration to Default"
        if os.path.isfile("/%s/default/wireless_default.ini"%weblib.PROOT):
            shutil.copyfile("/%s/default/wireless_default.ini"%weblib.PROOT,"/config/wireless_default.ini")
        shutil.copyfile("/config/wireless_default.ini","/config/wireless.ini")
    elif val == 'ip':
        result += "Setting ip configuration to Default"
        shutil.copyfile("/%s/etc/network/interfaces_dhcp"%weblib.PROOT,"/%s/etc/network/interfaces"%weblib.PROOT)
        weblib.pytouch("/%s/etc/random_ip"%weblib.PROOT)
    else:
        result += "Wrong argument, doing nothing\n"
    os.sync()
    result += "</BODY></HTML>"
    return render_template_string(result)

@app.route("/enableSecurity")
def enableSecurity():
    return render_template_string("<HTML><HEAD><meta http-equiv=\"refresh\" content=\"0;url=/administration?message=Security+enabled\"></HEAD></HTML>")

@app.route("/install_software", methods=['POST'])
def installSoftware():
    success = False
    print("main route")
    # Check if sd-card are mounted on unit
    with open("/proc/mounts") as txt:
        if '/dev/mmcblk0p1' in txt.read():
            print("in dev/mmcblk")
            success = True
    folder = "tmp/install"
    if success:
        folder = "data/install"
    os.makedirs("/%s/"%folder, exist_ok=True)
    weblib.removefile("/%s/install_package.tar"%folder)
    weblib.removefile("/%s/install_package.tgz"%folder)
    try:
        shutil.rmtree("/%s/install_package"%folder)
    except:
        pass
    file = request.files['filename']
    if file:
        file.save("/%s/install_package.tgz"%(folder))
        os.sync()
        # It's an uploaded file
        # strip leading path from file name to avoid directory traversal attacks
        subprocess.call(["gunzip","/%s/install_package.tgz"%folder])
        os.sync()
        subprocess.call(["tar","xf","/%s/install_package.tar"%folder,"--directory","/%s/"%folder])
        os.sync()
        subprocess.call(["/bin/sh","/%s/install_package/install_arm.sh"%folder])
        weblib.removefile("/%s/install_package.tar"%folder)
        weblib.removefile("/%s/"%folder)
        try:
            shutil.rmtree("/%s/install_package"%folder)
        except:
            pass
    weblib.removefile("/%s/[0-9]*.chunk"%folder)
    return render_template_string("""<html>
            <head>
                <meta http-equiv=\"refresh\" content=\"4;url=/administration\">
            </head>
            <body class =\"mainBody\">
                <br>
                <h2>The software were successfully installed.</h2>
                <p>Reboot the 900-unit after all installations are completed.</p>
            </body>
        </html>""")


@app.route("/reboot", methods=['POST'])
def reboot():
    try:
        shutil.copy("/tmp/wireless_telegrams_*","/config/tmp/")
    except:
        pass
    os.sync()
    subprocess.call(["/usr/bin/killall","S99crond"])
    subprocess.call(["/sbin/reboot"])
    return render_template_string(
        """<html><body>
            System is rebooting.<br />
            Please wait a few minutes and then push the refresh button.
            </body></html>""")





@app.route("/setClock", methods=['POST'])
def setClock():
    try:
        new_date = request.json["TimeDate"]
        date, t = tuple(new_date.split(' '))
    except Exception as e:
        print("Failed to get TimeDate from request", str(e))
        return Response(status=400)
    try:
        set_date_and_time(date,t)
    except Exception as e:
        print("Failed to set date" , str(e))
        return Response(status=400)

    return Response(status=200)

def set_ping_reboot(ping_interval, ping_address):
    CRON_PING_REBOOT_TMP_FILE = "%s/etc/cron_ping_reboot.tmp"%weblib.PROOT
    print("set ping rebbot" , ping_interval , ping_address)
    
    tmpstr = ""
    if ping_interval == 'Unset':
        weblib.removefile("%s/etc/cron_ping_reboot"%weblib.PROOT)
    else:
        tmpstr = ""
        if ping_interval == '30m':
            tmpstr = "27,57 * * * *"
        elif ping_interval == '2h':
            tmpstr = "57 */2 * *"
        elif ping_interval == '6h':
            tmpstr = "57 */6 * *"
        elif ping_interval == '24h':
            tmpstr = "57 0 * *"
            
        if len(tmpstr) > 0:
            tmpstr += " /etc/checkIFStatus.sh eth0 ping %s\n" % ping_address
            with open(CRON_PING_REBOOT_TMP_FILE, 'w') as tmp:
                tmp.write(tmpstr)
            os.replace(CRON_PING_REBOOT_TMP_FILE, "%s/etc/cron_ping_reboot" % weblib.PROOT)
            os.sync()
    
    cnf = configparser.ConfigParser()
    cnf.read("/config/pi900_system.ini")
    if 'PingReboot' in cnf.sections():
        cnf['PingReboot']['ping_interval'] = ping_interval
        cnf['PingReboot']['ping_address'] = ping_address
    else:
        cnf['PingReboot'] = {
            'ping_interval': ping_interval,
            'ping_address': ping_address,
        }

    with open('/config/pi900_system.ini','w') as f:
        cnf.write(f)

@app.route("/setEthernet",methods=['POST'])
def setEthernet():
    INTERFACES_TMP_SAVE_FILE = "%s/etc/interfaces.tmp"%weblib.PROOT
    errors = []

    # get ping interval
    try:
        ping_interval = request.form['ping_interval']
    except Exception:
        ping_interval = 'Unset'

    # get ping address
    try:
        ping_address = request.form['ping_address']
        ip_address(ping_address)
    except Exception:
        if ping_interval != 'Unset':
            return render_template('fragments/_errors_list.html',errors=['Invalid ping address'])
        ping_address = ""
        
    try:
        ethernet1Type = request.form["ethernet-type"]
    except Exception as e:
        app.logger.exception("failed to get ethernet type", exc_info=e )
        ethernet1Type = 'D'
    

    @after_this_request
    def add_header(response):
        if response.status_code == 200:
            set_ping_reboot(ping_interval, ping_address)
            if os.path.isfile(INTERFACES_TMP_SAVE_FILE):
                dest = "/%s/etc/network/interfaces" % weblib.PROOT
                os.replace(INTERFACES_TMP_SAVE_FILE, dest)
                print("Saving ethernet settings", dest)
                try:
                    os.sync()
                    weblib.removefile(INTERFACES_TMP_SAVE_FILE)
                    weblib.removefile("%s/etc/cron_ping_reboot.tmp" % weblib.PROOT )
                    subprocess.call(["reboot"])
                except Exception as e:
                    print("failed to save ethernet settings", str(e))
        
        return response
    
    if ( ethernet1Type == 'D'): # Dynamic IP2
        shutil.copyfile("/%s/etc/network/interfaces_dhcp"%weblib.PROOT,INTERFACES_TMP_SAVE_FILE)

        try:
            rand_ip = request.form["setrand"].lower() == 'set'
        except Exception:
            rand_ip = False

        
        piigab_system.set_randomized_ip(rand_ip, "/%s/etc/random_ip" % weblib.PROOT)
        
        return "<p>Changed to DHCP ( Dynamic IP address ). Type in correct IP-address and refresh browser</p>"
    
    # Static ip
    piigab_system.set_randomized_ip(False, "/%s/etc/random_ip" % weblib.PROOT)
    # validate ip-address
    try:
        ip_addr = request.form['IP1']
        ip_address(ip_addr)
    except Exception:
        errors.append("Invalid ip-address")
    
    # validate subnetmask
    try:
        subnet_mask = request.form['SM1']
        ip_address(subnet_mask)
    except Exception:
        errors.append("Invalid subnet address")

    # validate gateway address
    try:
        gateway_addr = request.form['GW1']
        ip_address(gateway_addr)
    except Exception:
        errors.append("Invalid gateway address")

    # validate nameserver
    try:
        nameserver = request.form['NSERV']
        if len(nameserver) == 0:
            nameserver = "8.8.8.8" # if nothing is set use google ns
        else:
            ip_address(nameserver)
    except Exception:
        errors.append("Invalid nameserver address")
    

    if len(errors) > 0:
        app.logger.error(errors)
        return render_template('fragments/_errors_list.html',errors=errors)

    # save configuration

    tmpstr  = "auto lo\niface lo inet loopback\nauto eth0\n"
    tmpstr += "iface eth0 inet static\n"
    tmpstr += "\taddress %s\n" % ip_addr
    tmpstr += "\tnetmask %s\n" % subnet_mask
    tmpstr += "\tgateway %s\n" % gateway_addr
    
    weblib.removefile("/etc/resolv.conf") # In case this is a symlink
    with open("/%s/etc/resolv.conf" % weblib.PROOT, "w") as f:
        f.write("nameserver %s\n" % nameserver)
    if weblib.PROOT != "":
        os.symlink("/%s/etc/resolv.conf" % weblib.PROOT,"/etc/resolv.conf")

    with open(INTERFACES_TMP_SAVE_FILE, 'w') as tmp:
        tmp.write(tmpstr)


    return "<p>IP address changed. Type in correct IP-address and refresh browser.</p>"



@app.route("/mbushub_setfailsafe", methods=['GET', 'POST'])
def mbushub_SetFailsafe():
    try:
        slnr=int(request.args.get("SLNR"))
    except:
        pass
    urlpart="mbushub_portconf?SLNR=%s"%slnr
    onoff = weblib.getParameter("rs485_failsafe")
    weblib.setGPIO("pioA24",onoff)
    if (onoff == 'On'):
        weblib.pytouch("/config/rs485_failsafe")
    else:
        weblib.removefile("/config/rs485_failsafe")
    os.sync()

    return render_template_string("""<HTML>\n<HEAD>
            <meta http-equiv=\"refresh\" content=\"2;url=/{{content}}\">
        </HEAD>\n<BODY>
            Setting RS-485 Failsafe
        </BODY>\n</HTML>""", content=urlpart)


@app.route("/identity", methods=['POST']) # TODO move me to system api
def set_identity():
    try:
        identification = request.json['identity']
        identification = identification.encode('utf-8')
        identification = identification.decode('utf-8')
        try:
            with open("/var/www1/identification.py", "w") as file:
                file.write(identification)
                return Response(status=200)
        except Exception:
            print("failed to write to identity file")
            return Response(status=400)
    except Exception as e:
        print("failed to set identity", str(e))
        return Response(status=400)


@app.route("/setLicense", methods=['GET'])
def setLicense():
    lstr=""
    retstr=""
    if request.args.get("licstr"):
        lstr=request.args.get("licstr")
        lstr = lstr.split(" ")[0]
    ret = None
    try:
        p1 = subprocess.Popen(["/binary/mbushub.elf", "-c", lstr],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            stdin=subprocess.PIPE)
        (p1out,p1err)=p1.communicate()
        print("rc",p1err.decode('utf-8'))
        print("stdout",p1.returncode)
        print("stderr", p1out.decode('utf-8'))
    except Exception as e:
        print("lic err:", str(e))
    if p1.returncode != 0:
        retstr="Incorrect license string. No change made."
    else:
        try:
            cf=open('/config/mbushub_lic.ini','w')
            cf.write("[Main]\n")
            cf.write("licstr=%s\n"%lstr)
            cf.close()
            os.sync()
            subprocess.call(["/%s/etc/init.d/S94mbushub"%weblib.PROOT,"restart"])
            #subprocess.call(["/%s/etc/init.d/S49cyclic"%weblib.PROOT,"restart"])
        except:
            pass
        retstr="Setting key to: %s"%lstr
    return render_template_string("""Content-Type: text/html\n
        <HTML>\n<HEAD>
        <meta http-equiv=\"refresh\" content=\"2;url=/basic_settings\">
        </HEAD>\n<BODY>{{content}}</BODY>\n</HTML>""", content=retstr)


def is_valid_hostname(hostname):
    if hostname[:-1] == ".":
        # strip exactly one dot from the right, if present
        hostname = hostname[:-1]
    if len(hostname) > 253:
        return False

    labels = hostname.split(".")
    # dont allow single words
    if len(labels) == 1:
        return False
    # the TLD must be not all-numeric
    if re.match(r"[0-9]+$", labels[-1]):
        return False

    allowed = re.compile(r"(?!-)[a-z0-9-]{1,63}(?<!-)$", re.IGNORECASE)
    return all(allowed.match(label) for label in labels)

def is_valid_ip(ipaddr):
    try:
        ip_address(ipaddr)
        return True
    except ValueError:
        return False

@app.route("/administration_setntp", methods=['POST'])
def administration_SetNTP():
    error = False
    ntpip=weblib.getParameter("ntpIP")
    is_hostname = is_valid_hostname(ntpip)
    is_ip_addr = is_valid_ip(ntpip)
    disable_ntpd = len(ntpip) == 0
    is_ntp_valid = is_hostname or is_ip_addr or disable_ntpd
    config = """server %s iburst
server 0.pool.ntp.org iburst
server 1.pool.ntp.org iburst
server 2.pool.ntp.org iburst
server 3.pool.ntp.org iburst

# Allow only time queries, at a limited rate, sending KoD when in excess.
# Allow all local queries (IPv4, IPv6)
restrict default nomodify nopeer noquery limited kod
restrict 127.0.0.1
restrict [::1]\n""" % ntpip

    if disable_ntpd:
        weblib.removefile("%s/default/ntpd"%weblib.PROOT)
        config = ""
    if is_ntp_valid:
        f=open("%s/etc/ntp.conf" % weblib.PROOT , 'w')
        f.write(config)
        f.flush()
        f.close()
        os.sync()

        try:
            subprocess.call(["/etc/init.d/S49ntp","restart"])
        except:
            pass
    else:
        error = True
    retstr=""
    if error:
        retstr="Incorrect Address"
    else:
        retstr="Setting NTP to address, %s"%ntpip
    return render_template_string("""<HTML>\n<HEAD>
            <meta http-equiv=\"refresh\" content=\"2;url=/administration\">
        </HEAD>\n<BODY>\n{{content}}\n</BODY>\n</HTML>""",content=retstr)


def get_timezone():
    """returns the selected timezone. Returns Europe/Stockholm in case of error."""
    try:
        with open(ZONEINFO_FILE,"r") as f:
            zone_data = json.load(fp=f)
            return zone_data['zone']
    except Exception as e:
        print("failed to read zoneinfo no file found", str(e))
        return "Europe/Stockholm" # FIXME. Remove when OS-2016 is obsolete.

def set_timezone(new_zone):
    """returns the selected timezone"""
    result_list = list(filter(lambda x: x[0] == new_zone , get_timezone_data() ))
    if len(result_list) == 1:
        posix_format = result_list[0][1]
        zone = result_list[0][0]

        with open(ZONEINFO_FILE,"w") as f:
            json.dump({
                "zone":zone
            },fp=f)
        with open('/etc/TZ',"w") as f:
            print("saving timezone:", posix_format)
            f.write(posix_format+"\n")
            os.environ["TZ"] = posix_format
            time.tzset()
            return
    else:
        raise ValueError("Invalid Timezone selected")


@app.route('/available-timezones', methods=['GET'])
def get_available_timezones():
    return jsonify(list(sorted(map(lambda x: x[0], get_timezone_data()))))

@app.route("/selected-timezone", methods=['POST','GET'])
def timeZone():
    if request.method == 'POST':
        try:
            selected = request.json["selected_timezone"]
            print("Timezone: ", selected)
            set_timezone(selected)
            return Response(status=200)
        except Exception as e:
            print("failed to set timezone: ", e)
            return Response(status=400)
    if request.method == "GET":
        timezone = get_timezone()
        return jsonify({"selected_timezone": timezone})


def get_timezone_data():
    "returns a list of tuples that holds two strings (zone, posix format)"
    TZDATA_PATH = "tzdata.txt"
    try:
        data = []
        with open(TZDATA_PATH, 'r') as tzdata:
            for line in tzdata:
                data.append(tuple(line.split('\t')))
        return data
    except Exception as e:
        print("failed to read timezone file: ", TZDATA_PATH, str(e))
    return []

@app.route("/setWebConf", methods=['POST'])
def setWebConf():
    try:
        https_val=request.json['selected']
    except Exception as e:
        https_val = ""

    @after_this_request
    def restart_server(response):
        if response.status_code == 200:
            print("Restarting lighttpd server")
            subprocess.call(["/etc/init.d/S50lighttpd","restart"])
            os.sync()
        else:
            print("Response status was not 200, not restarting server")
        return response


    if ( https_val in ['https', 'http'] ):
        lightttpd_conf = "/etc/lighttpd/lighttpd.conf"

        http_conf = "/etc/lighttpd/lighttpd_http.conf"
        https_conf = "/etc/lighttpd/lighttpd_https.conf"

        weblib.removefile(lightttpd_conf)
        if https_val == "http":
            os.symlink(http_conf, lightttpd_conf )
        elif https_val == "https":
            os.symlink(https_conf, lightttpd_conf )

        return jsonify({"message":"Server setting changed to: " + str(https_val) + ", Restarting server... Please wait a few seconds and then refresh your browser. "})

    return Response("invalid request", status=400)

@app.route("/showFile")
def showFile():
    cstr=""
    try:
        fname="/config/" + secure_filename(os.path.basename(request.args.get('FILE')))
        f=open(fname,'r')
        cstr=f.read()
        f.close()
    except:
        cstr="No Filename given"
    cstr=cstr.replace('\n','<br>')
    hstr="<HTML>\n<HEAD>\n</HEAD>\n<body>%s</body>\n</HTML>\n"%cstr
    return render_template_string(hstr)

@app.route("/showCsvFile")
def showCsvFile():
    str0 = "\n\n<html>\n<head>\n<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\"/>\n</head>\n<body>\n<table border=1>\n<tr>\n<td>"
    try:
        fname='/config/'+secure_filename(os.path.basename(request.args.get('FILE')))
        with open(fname,mode='r',encoding='utf-8',errors='replace') as file:
            a = file.read()
            for i in range(0,37):
                str0 += "%d</td><td>"%i
            str0 += "</td>\n</tr>\n<tr>\n<td>"
            str0 += "%s"%a.replace(',','</td><td>').replace('\r','').replace('\n','</td>\n</tr>\n<tr>\n<td>').replace('<td></td>','<td>&nbsp;</td>')
            str0 += "</td>\n</tr>\n</body>\n</html>"
            return render_template_string(str0)

    except Exception as e:
        str0 += "<h1>Failed to show csv file. </h1> </br> <p>" + str(e) + "</p>"
        return render_template_string(str0)

@app.route("/cgi-bin/showCsvFile.py") # Old route Remove when not used anymore.
def showCsvFile1():
    fname='/config/'+secure_filename(os.path.basename(request.args.get('FILE')))
    f=open(fname,encoding="utf-8")
    a=f.read()
    f.close()
    str0 = "\n\n<html>\n<head>\n<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\"/>\n</head>\n<body>\n<table border=1>\n<tr>\n<td>"
    for i in range(0,37):
        str0 += "%d</td><td>"%i
    str0 += "</td>\n</tr>\n<tr>\n<td>"
    str0 += "%s"%a.replace(',','</td><td>').replace('\r','').replace('\n','</td>\n</tr>\n<tr>\n<td>').replace('<td></td>','<td>&nbsp;</td>')
    str0 += "</td>\n</tr>\n</body>\n</html>"
    return render_template_string(str0)

@app.route("/showBrowseCsvFile")
def showBrowseCsvFile():
    fname='/tmp/'+secure_filename(os.path.basename(request.args.get('FILE')))
    f=open(fname,encoding="utf-8")
    a=f.read()
    f.close()
    str0 = "\n\n<html>\n<head>\n<meta charset=\"UTF-8\">\n</head>\n<body>\n<table border=1>\n<tr>\n<td>"
    for i in range(0,37):
        str0 += "%d</td><td>"%i
    str0 += "</td>\n</tr>\n<tr>\n<td>"
    str0 += "%s"%a.replace(',','</td><td>').replace('\r','').replace('\n','</td>\n</tr>\n<tr>\n<td>').replace('<td></td>','<td>&nbsp;</td>')
    str0 += "</td>\n</tr>\n</body>\n</html>"
    return render_template_string(str0)


@app.route("/uploadBackupConf", methods=['POST'])
def uploadBackupConf():
    file = request.files['bconffile']
    
    temp_dir = "/tmp/backups"
    
    os.makedirs(temp_dir,exist_ok=True)
    
    tgzfile = temp_dir+"/pi%sconfig_backup.tgz"% str(weblib.pi900_type)
    
    if os.path.exists(tgzfile):
        os.remove(tgzfile)
    
    if file == None:
        return render_template_string("""<HTML><HEAD>
        <meta http-equiv=\"refresh\" content=\"0;url=/administration\">
        </HEAD><body><h1>Failed to Backup, no backup file was uploaded</h1></body></HTML>""")
    
    file.save(tgzfile)
    try:
        piigab_system.extract_and_save_backup(tgzfile, weblib.PROOT + '/config/',temp_dir=temp_dir)
    except Exception:
        traceback.print_exc()
        return render_template_string("""<HTML><HEAD>
        <meta http-equiv=\"refresh\" content=\"0;url=/administration\">
        </HEAD><body><h1>Failed to Backup</h1></body></HTML>""")
    finally: # cleanup
        shutil.rmtree(temp_dir,ignore_errors=True) # make sure extracted files has been removed. in case exception is thrown
    
    return render_template_string("""<HTML><HEAD>
            <meta http-equiv=\"refresh\" content=\"0;url=/administration\">
            </HEAD><body><h1>Successfully uploaded backup!</h1></body></HTML>""")

@app.route("/uploadCSV", methods=['GET','POST'])
def uploadCSV():
    file = request.files['conffile']
    if file and allowed_file(file.filename):
        filename = secure_filename(file.filename)
        file.save(os.path.join("/config", filename))
    return render_template_string("""<HTML><HEAD>
            <meta http-equiv=\"refresh\" content=\"0;url=/administration\">
            </HEAD></HTML>""")

@app.route("/uploadCSVToOverview", methods=['GET','POST'])
def uploadCSVToOverview():
    file = request.files['conffile']
    if file and allowed_file(file.filename):
        filename = secure_filename(file.filename)
        file.save(os.path.join("/config", filename))
    return render_template_string("""<HTML><HEAD>
            <meta http-equiv=\"refresh\" content=\"0;url=/overview\">
            </HEAD></HTML>""")


@app.route("/cache.js", methods=['GET'])
def file_cache():
    """this is loaded in index.html use this javascript object when
    fetching resources in js /manufacturers?hash=myhashfromcachefile,
    for manufacturer list and devicetypes currently. might be added upon later.
    you will always be able to reach file_hashes variable in javascript since it is loaded first in index.html
    so file_hashes.manufacturers to get that hash
    """
    with open('static/data/manufacturers.txt', mode='rb') as f:
        sha = hashlib.sha256()
        sha.update(f.read())
        manufacturers = sha.hexdigest()

    with open('static/data/device_types.json', mode='rb') as f:
        sha = hashlib.sha256()
        sha.update(f.read())
        devices = sha.hexdigest()

    blob = """const file_hashes = {{
        'manufacturers': '{}',
        'device_types': '{}',
    }}""".format(manufacturers, devices)

    return Response(blob, mimetype='application/javascript')


@app.route('/ethernet-settings',methods=['GET'])
def get_ethernet_settings():
    """ based on the query parameter ethernet-type we will either return 
        A form for setting dynamic ip. or a form for static ip. """
    eth_type = request.args.get('ethernet-type')
    eth_info = piigab_system.get_interfaces_info()
    if eth_type is None:
        app.logger.error("No ethernet type argument found. expected 'D' or 'S' dynamic/static")
        return Response(400)
    if eth_type == 'S':
        return render_template('fragments/_ethernet_static_form.html', **eth_info)
    elif eth_type == 'D':
        eth_info.update({'randip': piigab_system.is_randomized_ip()})
        return render_template('fragments/_ethernet_dynamic_form.html', **eth_info)
    
    app.logger.error("Unandled case in /ethernet-settings")
    return Response(500)

@app.route('/test-connection', methods=['GET'])
def test_ping_connection():
    try:
        ip = request.args['ip']
    except Exception:
        return Response(status=400)
    try:
        ip_address(ip)
        ip_ok = True
    except Exception:
        print("invalid ip")
        ip_ok = False

    try:
        hostname_ok = is_valid_hostname(ip)
    except Exception:
        hostname_ok = False
    
    if not hostname_ok and not ip_ok:
        print("invalid ip and hostname")
        return Response(status=400)

    return_code = subprocess.call(['ping','-c','4', ip],timeout=6)
    if return_code != 0:
        print("failed to ping",ip)
        return Response(status=400)
    
    return Response(status=200)

if __name__ == "__main__":
    if os.environ['FLASK_ENV'] == 'development':
        app.run(use_debugger=True, debug=True, use_reloader=False, host='0.0.0.0',threaded=True)
