#!/usr/bin/python3
# -*- coding: utf-8 -*-;
#kate: space-indent on; indent-width 4; mixedindent off; indent-mode python

import struct, sys, socket,  csv
import time,  datetime
from weblib import *
"""
The program works on meterlist files with the following format:
0   , 1    , 2   , 3   , 4          , 5           , 6               , 7                  , 8                , 9
<Id>,<MFCT>,<VER>,<MED>,<MFCT STRNG>,<Fabrication>,<Primary Address>, <Verified or empty>,<Meter Type W,R,N>, <timestamp>

The meterlist is named meterlist.txt and is placed in either /tmp/ or /config/
The Id number must be there. The other are optional for the verification.

The /tmp directory (RAM memory, doesn't age flash memory) is for preconfiguring.
The /config directory is for files that have been worked on by the user and they should be treated with
respect.
Respect means :)
Do not overwrite without the users consent!

The list of searched meters is verified by definition.
An uploaded list must be verified and MFCT,VER,MED must be added to it.

A meter without MFCT,VER,MED must not be appended to the /config/... file.

ToDo:
The /config/meterlist.csv must be verified as well.
"""

def PrintData(data):
    tstr = ""
    for j in range(len(data)):
        if j%16 == 0:
            tstr += '\n'
        tstr += "%02X "%data[j]
    print(tstr)

def PrintDataOnOneLine(data):
    tstr = ""
    for j in range(len(data)):
        tstr += "%02X "%data[j]
    return tstr
    #print(tstr, end="")

def CalcBcc(arr):
    bcc=0
    for i in range(len(arr)):
        bcc += arr[i]
    return bcc & 0xFF


def MfctToStr(mfct):
    if mfct == 0xFFFF:
        return ""
    strng  = "%c"%(((mfct>>10)&0x1F)+64)
    strng += "%c"%(((mfct>>5)&0x1F)+64)
    strng += "%c"%((mfct&0x1F)+64) # Manufacturer number
    return strng

def CreateShortFrame(cfield, afield):
    req   = b"\x10"+struct.pack("BB",cfield, afield)
    req += struct.pack("B",CalcBcc(req[1:])) + b"\x16"
    return req

def CreateSlvSelect(idnum, mfct, ver, med, fab = None):
    req1 = b"\x53\xFD\x52" # (C-field),FD is the primary address 253, 52 is the selection of slaves (CI-field)
    req1 += struct.pack("<IHBB",idnum, mfct, ver, med)
    if fab:
        req1 += b"\x0C\x78" + struct.pack("<I", fab)
    req = b"\x68" + struct.pack("BB",len(req1),len(req1)) + b"\x68" + req1 # Length bytes between 68 L-byte L-byte 68 is => length (in bytes) of secondary address(idnum(4bytes), manufacturer(2bytes),version(1byte),medium(1byte)) + 3bytes(C-field,A-field,CI-field)
    req += struct.pack('B',CalcBcc(req[4:]))+b"\x16"
    return req

def CreateReqUd2(adress):
    toggle = 0x7B
    checkSum = (toggle + adress) & 0xFF # address 253(integer) + 7B(hex) =  178(hex) => 178(hex) & FF = 78(hex)
    req_ud2 = struct.pack("BBBBB", 0x10, toggle, adress, checkSum, 0x16)
    return req_ud2

def read_csv_file(sfname, mode):
    srows = []
    try:
        cf=open(sfname,mode)
        lines = cf.readlines()
        cf.close()
        delimiterChar = ""
        for line in lines:
            if line[0] != '#':
                if ";" in line:
                    delimiterChar=";"
                break
        cf=open(sfname,mode)
        if delimiterChar == ";":
            sreader = csv.reader(cf,  delimiter=';')
        else:
            sreader = csv.reader(cf)
        srows=[]
        for row in sreader:
            if len(row) > 0:
                if row[0][0] != "#":
                    for i in range(len(row)):
                        row[i] = row[i].strip() # Strip unnecessary leading and trailing spaces
                    srows.append(row)
        cf.close()
    except:
        pass
    return srows

def VerifyLongFrame(frame):
    if len(frame) == 0:
        print("No response received")
        return False
    # Check number of received bytes.
    if len(frame) < 21:
        print("Too few bytes received")
        print(len(frame))
        return False

    # Check frame characters.
    if (frame[0] != 0x68 or frame[3] != 0x68):
        print("Incorrect frame characters")
        return False

    # Check length.
    if (frame[1] != frame[2] or frame[1] != len(frame)-6):
        print("Incorrect message length")
        return False
    # Check checksum.
    if (CalcBcc(frame[4:len(frame)-2]) != frame[len(frame)-2]):
        print("Incorrect checksum")
        return False
    return True

def isHex(i):
    try:
        int(i, 16)
        return True
    except:
        return False

cont = True # Set to false for the verfication to stop
def DoVerify(ip, port, init, meterfile):
    global cont
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    s.settimeout(5) # A timeout is needed for meters that don't respond
    nodeaddresses = [[0x4129,0x01,0x36], [0x1596,0x14,0x31]] # This should preferably be read from a configuration file.
    meterlist = read_csv_file(meterfile, 'r')

    meterlist1 = []
    for meter in meterlist:
        #print("meter %s"%meter)
        #Here we have to look at the fabrication number to know if we have to get timestamp from wireless_nodes_telegrams.txt file or similar file.
        #The fabrication number have to be added for the local wireless card in the 900-unit.
        #
        if meter[0][0] == "#":
            continue
        if not cont:
            break
        retval = ""
        ok = True
        id = int(meter[0], 16)
        mfct = 0xFFFF
        ver   = 0xFF
        med = 0xFF
        fab = None
        prim = ''
        metertype = ""
        if len(meter) > 1 and isHex(meter[1]):
            mfct = int(meter[1],  16)
        if len(meter) > 2 and isHex(meter[2]):
            ver = int(meter[2],  16)
        if len(meter) > 3 and isHex(meter[3]):
            med = int(meter[3],  16)
        if len(meter) > 5 and isHex(meter[5]):
            fab = int(meter[5],16)
        if len(meter) > 6 and isHex(meter[6]): # This is in fact a Decimal but all decimals are in fact hexes as well :)
            prim=int(meter[6])
        if len(meter) > 8:
            metertype = meter[8]
        req = CreateSlvSelect(id, mfct, ver, med, fab)
        PrintData(req)
        s.sendto(req, (ip, port))
        time.sleep(0.05)
        resp = b''
        try:
            resp =  s.recv(1024)
        except:
            pass
        # Check for none ACK response.
        if len(resp) == 0:
            print("No response for SND_NKE")
            retval = "No SND_NKE_RESPONSE"
            ok = False
        elif len(resp) != 1 or resp[0] != 0xE5:
            print("Invalid response for SND_NKE")
            retval = "Invalid SND_NKE_RESPONSE, %02X"%resp[0]
            ok = False
        if ok:
            req = CreateReqUd2(253)
            s.sendto(req, (ip, port))
            time.sleep(0.05)
            try:
                resp =  s.recv(1024)
            except:
                pass
            PrintData(resp)
            if not VerifyLongFrame(resp):
                retval = "Invalid REQ_UD2_RESPONSE"
                ok = False
            time.sleep(0.1)
            #deselect meter
            req = CreateShortFrame(0x40, 253)
            s.sendto(req, (ip, port))
            time.sleep(0.05)
            try:
                s.recv(1024)
            except:
                pass
            time.sleep(3) # We need to wait for a timeout of different nodes. Explain better GG XXXXXXXXX. The sleep is needed, a timeout is not enough
        m=['']*10
        if metertype not in ['R', 'N', 'W','H','M']:
            metertype = 'U' # It is preferable to set it as a wired meter. It will be overwritten later. Check XXXXXXXXXXXXXXXXX
        if ok:
            retval = "Verified"
            (id1, mfct1,  ver1,  med1) = struct.unpack("<IHBB",resp[7:15])
            m[0] = "%08X"%id1
            m[1] = "%04X"%mfct1
            m[2] = "%02X"%ver1
            m[3] = "%02X"%med1
            m[4] = "%s"%MfctToStr(mfct1)
            m[6] = "%d"%resp[5] #Primary address
            m[7] = "Ver"
            if [mfct1, ver1, med1] in nodeaddresses:
                m[8] = "N"
            else:
                m[8] = metertype
        else:
            m[0] = "%08X"%id
            m[1] = "%04X"%mfct
            m[2] = "%02X"%ver
            m[3] = "%02X"%med
            try:
                m[4] = "%s"%MfctToStr(mfct)
            except:
                pass
            try:
                m[6] = "%d"%prim #Primary address
            except:
                pass
            m[7] = ''
            if [mfct, ver, med] in nodeaddresses:
                m[8] = "N"
            else:
                m[8] = metertype
        if fab:
            m[5] = "%08X"%fab
        meterlist1.append(m)
    if cont: # If the verification has been stopped. Don't overwrite the meterlist.
        cf=open("/config/meterlist.csv",'w', newline='') # This should be changed to /config/ for config file verification
        writer = csv.writer(cf, delimiter=',', quotechar='\"', quoting=csv.QUOTE_MINIMAL)
        for m in meterlist1:
            writer.writerow(m)
        cf.close
        meterlist_to_cyclic_conf()

if __name__ == "__main__":
    if len(sys.argv) < 5:
        print("Usage %s <IP-address> <IP-Port> <init> <meterlist csv file>"%sys.argv[0])
        sys.exit(0)
    ip = sys.argv[1]
    port = int(sys.argv[2])
    init = sys.argv[3]
    meterfile = sys.argv[4]
    DoVerify(ip, port, init, meterfile)



