import base64
import hashlib
import hmac
import json
import time
import os
import shutil


PWHASH_BCRYPT = 0
PWHASH_PBKDF2 = 1

try:
    PWHASH_DEFAULT = PWHASH_BCRYPT
    PWHASH_WORK_FACTOR = 8
    import bcrypt
except:
    PWHASH_DEFAULT = PWHASH_PBKDF2
    PWHASH_ROUNDS = 10000

unreachable = False

class PasswordManager:
    """
    The password manager is responsible for checking that passwords match,
    updating passwords for users, and slowing down bruteforce attacks by
    incurring login delays on password faulure.

    There must be only ever be one instance of this class within the web
    interface thus one must not directly instantiate it outside of the
    given global `password_manager` instance.
    """

    algorithm = PWHASH_DEFAULT
    """The algorithm used for the password hash (fallback on pbkdf2)"""

    guard = 0
    """Time of the last failed login"""

    attempts = 0
    """Number of failed login attempts since the last successful attempt"""

    timeout = 3
    """Configured timeout on login failure"""
    """
    The following init function sets the password to Admin if everything fails.
    """
    def __init__(self, PROOT="/usr/local/"):
        
        self.PASSWORD_FILE = "%s/etc/shadow.json" % PROOT
        
        if not os.access(self.PASSWORD_FILE, os.W_OK | os.R_OK):
            #Location of the old password database
            INSECURE_PASSWORD_FILE = "%s/etc/lighttpd/lighttpd.user" % PROOT
            DEFAULT_PASSWORD_FILE = "%s/default/shadow.json" % PROOT

            if not os.access(DEFAULT_PASSWORD_FILE, os.W_OK | os.R_OK):
                
                db={'Admin':{'pass':self.new_password('Admin'),'default': True}}
                with open(DEFAULT_PASSWORD_FILE, 'w') as file:
                    json.dump(db, file)
                    os.sync()

            try:
                with open(INSECURE_PASSWORD_FILE) as file:
                    db = {}

                    for line in file:
                        user, old_password = line.strip(' \r\n').split(':')
                        db[user] = {
                            'pass': self.new_password(old_password),
                            'default': 'Admin' == old_password,
                        }

                with open(self.PASSWORD_FILE, 'w') as file:
                    json.dump(db, file)
                    os.sync()
                os.remove(INSECURE_PASSWORD_FILE)
            except:
                shutil.copyfile(DEFAULT_PASSWORD_FILE, self.PASSWORD_FILE)


    def load(self):
        with open(self.PASSWORD_FILE) as file:
            return json.load(file)

    def new_password(self, password):
        if self.algorithm == PWHASH_PBKDF2:
            salt = os.urandom(32)
            pwhash = hashlib.pbkdf2_hmac('sha256', bytes(password, encoding='utf-8'), salt, PWHASH_ROUNDS)

            salt = base64.b64encode(salt).decode('utf-8')
            pwhash = base64.b64encode(pwhash).decode('utf-8')
            return salt + "$" + pwhash
        elif self.algorithm == PWHASH_BCRYPT:
            salt = bcrypt.gensalt(PWHASH_WORK_FACTOR)
            return base64.b64encode(bcrypt.hashpw(bytes(password, 'utf-8'), salt)).decode('utf-8')
        else:
            assert unreachable

    def update(self, user, password):
        """
        Update the password for the given user.
        """
        db = self.load()

        db[user]['pass'] = self.new_password(password)
        db[user]['default'] = False

        with open(self.PASSWORD_FILE + "-tmp", 'w') as file:
            json.dump(db, file)

        os.sync()
        os.rename(self.PASSWORD_FILE + "-tmp", self.PASSWORD_FILE)

    def check(self, user, password):
        """
        Securely check that the given password matches that stored with
        the account.
        """

        account = self.load().get('Admin')
        if not account:
            return False

        pw = account['pass']

        if self.algorithm == PWHASH_PBKDF2:
            salt, pwhash = pw.split('$')

            salt = base64.b64decode(salt)
            pwhash = base64.b64decode(pwhash)

            new = hashlib.pbkdf2_hmac('sha256', bytes(password, 'utf-8'), salt, PWHASH_ROUNDS)

            return hmac.compare_digest(pwhash, new)
        else:
            auth = base64.b64decode(pw)
            return bcrypt.checkpw(bytes(password, 'utf-8'), auth)

    def has_default_password(self, user):
        """
        Check if the user is currently using their default password.
        """
        account = self.load().get(user)

        if not account:
            return False

        return account.get('default') == True


    def is_rate_limited(self):
        """
        Check the current request should be rate limited.
        """
        return time.time() < self.guard

    def delay(self, seconds):
        """
        Set a time for which login requests are not allowed.
        """
        self.timeout = seconds
        self.guard = time.time() + seconds

    def success(self):
        """
        Reset login attempts.
        """
        self.attempts = 0

    def fail(self):
        """
        Set a time for which login requests are not allowed after two failed login attempts.
        """
        if self.attempts > 0:
            self.delay(3)
        self.attempts += 1

