import json

def getHex(i, defval):
    retval = defval
    try:
        retval = int(i, 16)
    except:
        pass
    return retval


# Changes ['12345678','',None] to [0x12345678,0xFFFF,0xFF,0xFF]
def secAddrStrToHex(saddr):
    id = getHex(saddr[0], 0xFFFFFFFF)
    mfct = getHex(saddr[1], 0xFFFF)
    ver = getHex(saddr[2], 0xFF)
    med = getHex(saddr[3], 0xFF)
    return [id, mfct, ver, med]

# Only full wildcards are allowed in this function. Exempel mfct = 0x4129 or 0xFFFF. Not 0x41FF
# Id must be without a wildcard
# Only saddr2 is checked for wildcards
# Return values 0 if no match, 1 if id's match, 2 if all fields match, -1 in case of an error
def compareSecAddresses(saddr1, saddr2):
    match = 0
    id1 = saddr1[0]
    mfct1  = saddr1[1]
    ver1    = saddr1[2]
    med1  = saddr1[3]
    id2 = saddr2[0]
    mfct2  = saddr2[1]
    ver2    = saddr2[2]
    med2  = saddr2[3]
    if id1 == id2:
        match = 1
        print("match 1")
        print("mfct1 mfct2", mfct1,mfct2)
        if mfct1 == mfct2 and ver1 == ver2 and med1 == med2:
            print("match 2")
            match = 2
        elif mfct1 != mfct2 and mfct2 != 0xFFFF and mfct2 != "FFFF" and mfct2 != [None] and mfct2 != "":
            print("match 01")
            match = 0
        elif ver1 != ver2 and ver2 != 0xFF and ver2 != "FF" and ver2 != [None] and ver2 != "":
            print("match 02")
            match = 0
        elif med1 != med2 and med2 != 0xFF and med2 != "FF" and med2 != [None] and ver2 != "":
            print("match 03")
            match = 0
    return match

def wildcardEqual(a, b, size):
    for i in range(size * 2):
        mask = 0xf << (i * 4)
        if a & mask == mask: continue
        if b & mask == mask: continue
        if (a & mask) != (b & mask):
            return False
    return True

class SecondaryAddress:
    def __init__(self, secid, man, ver, med):
        self.id = secid
        self.manufacturer = man
        self.version = ver
        self.medium = med

    def __str__(self):
        return "%08X"%self.id + ".%04X"%self.manufacturer + ".%02X"%self.version + ".%02X"%self.medium

    def __eq__(self, other):
        if (
            not wildcardEqual(self.id, other.id, 4) or
            not wildcardEqual(self.manufacturer, other.manufacturer, 2) or
            not wildcardEqual(self.version, other.version, 1) or
            not wildcardEqual(self.medium, other.medium, 1)
            ):
            return False
        return True

    def toJson(self):
        return json.dumps(self.__dict__)

# Indexes
# | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
#
# User input 0x15FFFF62
# | 1 | 5 | F | F | F | F | 6 | 2 |

class SecondarySearch:
    def __init__(self, secondary):
        self.secondary = secondary
        # Frozen is a bit-mask of nibbles given by user.
        # I.e 15FFFF62 == 11000011      
        self.frozen = 0
        # Marker is the wildcard node with the highest index
        self.marker = 0
        self.lastIndex = 0

        lastIndexBeenSet = False
        for i in range(8):            
            mask = 0xf << (i * 4)
            if secondary.id & mask != mask:
                self.frozen |= 1 << i
            else:
                self.marker = i
                if not lastIndexBeenSet:
                    self.lastIndex = i
                    lastIndexBeenSet = True

        self.index = self.marker

    def __iter__(self):
        return self

    def __next__(self):
        value = self.next()
        if value:
            return value
        raise StopIteration

    # Checks if the current number is set by the user.
    def isFrozen(self):
        mask = 1 << self.index
        return self.frozen & mask == mask

    # "Hit" only walks downwards from index 7 to index 0    
    def hit(self):
        if self.index != 0:
            self.index -= 1
            while self.index != 0 and self.isFrozen():
                self.index -= 1

    def isEnd(self):
        offset = self.marker * 4
        marker = 0xa << offset
        mask = 0xf << offset
        return self.secondary.id & mask == marker

    # Checks if the current index is equal to the highest wildcard index.
    def isTop(self):
        return self.index == self.marker

    # Returns the value at current self.index
    def get(self):
        return (self.secondary.id >> (self.index * 4)) & 0xf

    # Set the given value at the current index.
    def set(self, value):
        value &= 0xf
        offset = self.index * 4
        mask = 0xf << offset
        self.secondary.id = (self.secondary.id & ~mask) | (value << offset)

    # "Next" only moves upward in the index from 0 to 7
    def next(self):
        if not self.secondary:
            return None

        while (self.get() == 0x9 or self.isFrozen()) and not self.isTop():
            if (not self.isFrozen()):
                self.set(0xf)
            self.index += 1

        self.set(self.get() + 1)
        if self.isEnd():
            self.secondary = None

        return self.secondary
