import subprocess
import configparser
import os
import weblib
import shutil
import traceback
import glob

"""
returns a dict of all the information about the license
raises exception if it fails to fetch that information from mbushub
"""

def get_license_information():
    out = subprocess.check_output(["/binary/mbushub.elf", "-S"])
    serialno = out.decode().strip("\n")

    out = subprocess.check_output(["/binary/mbushub.elf", "-l"])
    licensearray = out.decode().strip("\n").split(";")
    if len(licensearray) == 0:
        raise Exception("Invalid license")
    
    clients = licensearray[0]
    meters = licensearray[1]
    prot_trimmed_space = map( lambda x: str(x).strip() , licensearray[2].split(','))
    protocols = list(filter(lambda x: x != '-' and x != '',  prot_trimmed_space))
    cards = licensearray[3]
    return {
        "serial_number": serialno,
        "clients": int(clients.split(' ')[0]),
        "loads": int(meters.split(' ')[1]),
        "protocols": protocols,
        "cards": int(cards.lstrip(' ').split(' ')[0]),
        "license": get_license_string(),
        "is_license_overload":is_license_overload(),
    }


"""
returns the encrypted license string
"""


def get_license_string():
    hwconfig = configparser.RawConfigParser()
    hwconfig.read("/config/mbushub_lic.ini")
    return hwconfig.get("Main", "licstr")


def is_license_overload():
    try:
        conf = configparser.RawConfigParser()
        conf.read('/tmp/pi900_state.txt')
        error_flag = int(conf.get('Main','MBusErrorVal'))
    except Exception:
        error_flag = 0

    return (error_flag & 1) > 0 and (error_flag & 2) == 0


"""
updates the license string
raises exception if license is invalid
"""
def update_license(new_license):

    p1 = subprocess.Popen(["/binary/mbushub.elf", "-c", new_license],
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        stdin=subprocess.PIPE)
    (p1out,p1err)=p1.communicate()
    print("rc",p1err.decode('utf-8'))
    print("stdout",p1.returncode)
    print("stderr", p1out.decode('utf-8'))

    if p1.returncode != 0:
        raise Exception("Invalid License")

    with open('/config/mbushub_lic.ini','w') as conf:
        conf.write("[Main]\n")
        conf.write("licstr=%s\n"%new_license)
    os.sync()
    subprocess.call(["/%s/etc/init.d/S94mbushub"%weblib.PROOT,"restart"])

# returns general information about the gateway that you currently see on the start page.
def get_system_information():
    error_message = "No error"
    try:
        error_flag = 0

        conf = configparser.RawConfigParser()
        conf.read('/tmp/pi900_state.txt')
        board_temp = conf.get('Main','Temperature_40V')
        mbus_stage_temp = conf.get('Main','Temperature_MBus')
        error_flag  = int(conf.get('Main','MBusErrorVal'))

        if (error_flag & 1):
            error_message = "M-Bus Overload"
        if (error_flag & 2):
            error_message = "M-Bus Short Circuit"
        
        mbus_current      = float(conf.get('Main','MbusCurrent'))
        mbus_max_current   = float(conf.get('Main','MbusMaxCurrent'))
        
    except Exception:
        mbus_stage_temp = 0.0
        board_temp = 0.0
        mbus_current = 0.0
        error_flag = 0
        error_message = "No error"
    
    precentage_max_current = 0.0
    if mbus_max_current > 0:
        precentage_max_current = int(mbus_current/mbus_max_current * 100)

    return {
        "serial_number": weblib.serialNo(),
        "gateway_type": weblib.pi900_type,
        "os_name": weblib.PI900_OS_NAME,
        "os_version":weblib.pi900_os,
        "webinterface_version": weblib.pi900_version,
        "board_temp":float(board_temp),
        "mbus_stage_temp":float(mbus_stage_temp),
        "mbus_current": mbus_current,
        "error_flag": error_flag,
        "error_message": error_message,
        "percentage_used_of_max_current":precentage_max_current,
        "is_rs485_failsafe_on": weblib.getGPIO("pioA24") == 'On',
        "is_master_output_loop_1_on": weblib.getGPIO("pioA26") == 'On',
        "is_master_output_loop_2_on": weblib.getGPIO("pioA27") == 'On',
        "is_master_output_loop_3_on": weblib.getGPIO("pioA28") == 'On',
        "is_master_output_loop_4_on": weblib.getGPIO("pioA29") == 'On',
    }


def is_randomized_ip(path="/usr/local/etc/random_ip"):
    return os.path.isfile(path)

def set_randomized_ip(use_random_ip=True, path="/usr/local/etc/random_ip"):
    
    if use_random_ip:
        with open(path,'w') as f:
            f.close()
    else:
        try:
            os.remove(path)
        except OSError:
            pass # file does not exist
        except Exception:
            pass

def get_interfaces_info(filepath="/etc/network/interfaces"):
    """Parses interfaces file and returns a dict of the parsed info pretaining to eth0 and eth2"""
    
    eth_info = {
        'eth0GW':'',
        'eth2GW':'',
        'eth0TYPE':'',
        'eth0MAC' : '',
        'eth0IP':'',
        'eth0SM':'',
        'eth2TYPE':'',
        'eth2IP':"None",
        'eth2MAC':"None",
        'eth2SM':"None",
        'isEth0Static': False,
        'nameserv':'',
    }
    try:
        cf=os.popen("/bin/netstat -r -n")
        gwcs=cf.read().split('\n')
        cf.close()
        for line in gwcs:
            items = line.split()
            if len(items) > 0 and items[0] == '0.0.0.0':
                if items[7] == 'eth0':
                    eth_info['eth0GW'] = items[1]
                elif items[7] == 'eth1':
                    eth_info['eth2GW'] = items[1]
    except:
        pass

    try:
        cf=open("/etc/resolv.conf",'r')
        nwcs=cf.read().rstrip()
        cf.close()
        c=nwcs.split()
        if len(c)>0:
            if c[0]=='nameserver':
                eth_info['nameserv']=c[1]
    except:
        pass
    
    cf=open(filepath,'r')
    nwcs=cf.read().split('\n')
    cf.close()

    
    for i in nwcs:
        if not i=='':
            c=i.split()
            if len(c)>0:
                if c[0]=='iface' and c[1]=='eth0':
                    eth_info['eth0TYPE']=c[3]
                if c[0]=='iface' and c[1]=='eth1':
                    eth_info['eth2TYPE']=c[3]

    cf=os.popen("/sbin/ifconfig eth0 2>/dev/null")
    ifcs=cf.read().split('\n')
    cf.close()
    eth_info['isEth0Static'] = eth_info['eth0TYPE'] == 'static'
    eth_info['eth0MAC']=ifcs[0].split()[4]
    eth_info['eth0IP']=ifcs[1].split()[1].split('addr:')[1]
    eth_info['eth0SM']=ifcs[1].split()[3].split('Mask:')[1]

    return eth_info
    
"""
    Tries to find the current vpn interface name to PiiGAB Portal by looking 
    at the wireguard interface config
    
    returns localhost if no vpn address is found
"""
def get_vpn_interface_address(wireguard_dir='/usr/local/etc/msb-py'):
 # looking in msb-py path because that is where they save the vpn conf
    if os.path.exists(wireguard_dir):
        for entry in os.listdir(wireguard_dir):
            # search for wireguard configuration file
            if not entry.endswith('.conf'): continue
            # determine if it is a wiregauard file and not some other conf
            with open(wireguard_dir + '/' + entry,'r') as f:
                line = f.readline()
                if not len(line): continue
                if not line.startswith('[Interface]'): continue
                
                for line in f.readlines():
                    if line.startswith("Address = "):
                        return line[ 10 : line.find("/", 10, len(line))]                
    return "127.0.0.1"



def extract_and_save_backup(filepath, save_dir="/config/",temp_dir="/tmp/backups"):
    """Summary extracts a .tgz backup file and overwrites existing configuration files with this backup
    
    Parameters:
    filepath -- the .tgz filepath
    save_dir -- the configuration save location, default to /usr/local/config/ (charleton)
    temp_dir -- the temporary directory where the backup files will be extracted to and later cleaned up
    """
    tarfile = filepath
    if filepath.endswith('.tgz'):
        tarfile = tarfile[:len(tarfile)-3] + "tar"
    else:
        raise Exception('expected a filename with a ".tgz" extension')
    
    if os.path.exists(tarfile):
        os.remove(tarfile)
    
    subprocess.check_output(["gunzip", filepath ])
    
    if not os.path.exists(temp_dir):
        os.makedirs(temp_dir)
    
    subprocess.check_output(["tar","xf", tarfile ,"--directory", temp_dir])
    # assume first it is charleton backup file tree structure
    tmp_file = temp_dir + "/usr/local/config/"
    # set to marlene file structure if that path does not exist
    if not os.path.exists( tmp_file ): 
        tmp_file = temp_dir + "/config/" # marlene backup file tree structure
    
    if not os.path.exists(tmp_file):
        raise Exception("Could not find extracted backup files")
    
    files_with_suffiexs_to_not_overwrite = [
        "_lic.ini",
        "_default.ini",
    ]
    
    for filename in os.listdir(tmp_file):
        for suffux in files_with_suffiexs_to_not_overwrite:
            if filename.endswith(suffux):
                try:
                    os.remove(tmp_file + filename)
                except Exception:
                    traceback.print_exc()
            
    def copy(src,dest):
        try:
            shutil.copyfile(src,dest)
        except shutil.SameFileError:
            traceback.print_exc()
        except Exception:
            print(src, dest, " error: failed to copy")
            traceback.print_exc()
    
    os.makedirs(save_dir, exist_ok=True)
    
    os.makedirs(save_dir + "/ca-certificates", exist_ok=True)
    
    os.makedirs(save_dir + "/ssh", exist_ok=True)
    
    [ copy(f_p,save_dir + "/" + os.path.basename(f_p)) for f_p in glob.glob(tmp_file+"*.ini")]
    [ copy(f_p,save_dir + "/" + os.path.basename(f_p)) for f_p in glob.glob(tmp_file+"*.[cC][sS][vV]")]
    [ copy(f_p,save_dir + "/" + os.path.basename(f_p)) for f_p in glob.glob(tmp_file+"*.[xX][mM][lL]")]
    [ copy(f_p,save_dir + '/ca-certificates/' + os.path.basename(f_p)) for f_p in glob.glob(tmp_file+"ca-certificates/*")]
    [ copy(f_p,save_dir + '/ssh/'+ os.path.basename(f_p)) for f_p in glob.glob(tmp_file+"ssh/*")]

    shutil.rmtree(tmp_file, True)
    shutil.rmtree(temp_dir, True)