import subprocess
import os
import traceback

class OPKGListException(Exception):
    pass

class OPKGParseException(Exception):
    pass

class OPKGUpdateException(Exception):
    pass

class OPKGUpgradeException(Exception):
    pass

class OPKGInstallException(Exception):
    pass

class OPKGSignatureException(Exception):
    pass

class Package:
    def __init__(
        self,
        name,
        description,
        is_installed,
        version,
        next_version=None,
    ):
        self.name = name
        self.description = description
        self.is_installed = is_installed
        self.version = version
        self.next_version = next_version


# Public


def parse_packages(opkg_output: str):
    packages = []
    for line in  opkg_output.splitlines():
        if len(str(line).lstrip()) == 0:
            continue
        try:
            packages.append(parse_package_from_line(line))
        except OPKGParseException:
            pass
    
    return list(filter(lambda x: x != None, packages))


def parse_package_from_line(line):
    split_lines = line.split(" - ")
    description = None
    if len(split_lines) == 2:
        name, version = tuple(split_lines)
    elif len(split_lines) == 3:
        name, version, description = tuple(split_lines)
    else:
        raise OPKGParseException("invalid opkg line: %s %s" % (line, len(line)))
    
    return Package(
        name=name,
        version=version,
        is_installed=False,
        description=description,
        next_version=None,
    )


def install_package(package_name):
    package_name = os.path.basename(package_name) # disallow offline install of ipk
    proc = subprocess.run(['opkg', 'install', package_name])
    if proc.returncode != 0:
        raise OPKGInstallException("failed to install package", package_name)
    print("successfully installed", package_name)

def upgrade_package(package_name):
    package_name = os.path.basename(package_name) # disallow offline upgrade of ipk
    update()
    package = get_package_info(package_name) # old package info
    subprocess.run(['opkg', 'upgrade', package_name])
    new_package = get_package_info(package_name)
    if new_package.version == package.version:
        raise OPKGUpgradeException("Failed to upgrade package, version is unchanged.")
    print("successfully upgraded", package_name)

"""
def install_bundle():
    opkg_verify_bundle = '/usr/local/bin/opkg-verify-bundle.sh'
    if not os.path.isfile(opkg_verify_bundle):
        raise OPKGInstallException(opkg_verify_bundle + " verify script not found")
    proc = subprocess.run([opkg_verify_bundle])
    if proc.returncode != 0:
        raise OPKGInstallException("failed to verify bundle")
    
    # install all packages in bundle

    # filter for ipks and install

    path = '/tmp/package-bundle'
    if not os.path.exists(path):
        raise OPKGInstallException("package bundle not found")
    
    install_results = []
    for ipk_file in filter(lambda x: str(x).endswith('.ipk') , os.listdir(path)):
        proc = subprocess.run(['opkg','install','--force-reinstall','--autoremove', path + "/" + ipk_file])
        result = {"package":os.path.basename(ipk_file),"success": False}
        if proc.returncode > 0:
            install_results.append(result)
            print("ERROR: failed to install", os.path.basename(ipk_file))
        else:
            result['success'] = True
            install_results.append(result)
        
    return install_results
"""

"""Installs a ipk file with opkg package manager, raises an error if install failed
Keyword arguments:
ipk_filepath -- location of the ipk file
force_reinstall -- forces a reinstall of your current version or upgrades if its a newer version.
Return: None
Raises exception of install failure
"""
def install_ipk(ipk_filepath, force_reinstall=False):
    cmd = ['opkg','install',ipk_filepath]

    if force_reinstall:
        cmd.append('--force-reinstall')
    try:
        subprocess.check_output(cmd)
    except Exception as e:
        traceback.print_exc()
        raise e

def update():
    opkg__update_list_script = '/usr/local/bin/opkg-update-list.sh'
    if os.path.isfile(opkg__update_list_script):
        print("Downloading packages & verifying signature...")
        proc = subprocess.run([ opkg__update_list_script ], capture_output=True)
        
        print(proc.stderr.decode(), proc.stdout.decode())
        if proc.returncode != 0:
            raise OPKGSignatureException(proc.stdout.decode())
        print("Signature valid")
        return
    raise OPKGSignatureException("Signature script not found!")

def list_packages():
    all_packages = parse_packages(_opkg_list_all())
    installed_packages = parse_packages(_opkg_list_installed())
    return _list_packages(installed_packages, all_packages)


# Private
def _opkg_list_installed():
    proc = subprocess.run(["opkg", "list-installed"], capture_output=True)

    if proc.returncode != 0:
        raise OPKGListException(
            "Failed to list installed packages when calling opkg list"
        )
    return proc.stdout.decode()

def _opkg_list_all():
    proc = subprocess.run(["opkg", "list"], capture_output=True)
    if proc.returncode != 0:
        raise OPKGListException(
            "Failed to list installed packages when calling opkg list"
        )
    return proc.stdout.decode()


def _list_packages(installed_packages, all_packages):
    seen_packages = {}

    # go over all installed packages first
    for installed in installed_packages:
        installed.next_version = None
        installed.is_installed = True
        seen_packages[installed.name] = installed

    # go over every package
    for package in all_packages:
        if package.name in seen_packages:
            if not seen_packages[package.name].description:
                seen_packages[package.name].description = package.description
            
            if seen_packages[package.name].version != package.version:
                seen_packages[package.name].next_version =  package.version
        else:
            package.next_version = package.version
            package.version = None
            seen_packages[package.name] = package
    return list(seen_packages.values())


def _parse_package_info(opkg_info_str:str):
    temp = opkg_info_str.splitlines(False)
    i = 0
    package = Package(
        name="",
        description="",
        is_installed=False,
        version="",
    )

    while enumerate(temp):
        if i >= len(temp)-1:
            break
        kv = temp[i].split(': ')
        key = kv[0].lower()
        if len(kv) == 2:
            value = kv[1]
        else:
            value = ""
        if key == 'package':
            package.name = value
        elif key == 'version':
            package.version = value
        elif key == 'description':
            description = True
            package.description = value
        elif key == 'status':
            package.is_installed = 'installed' in value.split(' ')
        else:
            pass
        i+=1
    return package

def get_package_info(program_name):
    proc = subprocess.run(['opkg','info', program_name],capture_output=True)
    if proc.returncode != 0:
        raise Exception("Could not get info from package ", str(proc.stderr.decode()))    
    return _parse_package_info(proc.stdout.decode())
