#!/usr/bin/python3

import os
import sys
import socket
import struct
import ipaddress
import datetime
import argparse
from enum import Enum
#import msvcrt

# Version.
major = 1
minor = 0
patch = 0
build = 0
version = "%d.%02d.%02d"%(major, minor, build)

# Title.
print("")
print(" =======================================================================")
print(" === PiiGAB M-Bus Browse V%s ================= Copyright PiiGAB ==="%(version))
print(" =======================================================================")
print("")

# =============================================================================
# === Time limit check ========================================================
# =============================================================================
timeLimit = False    # Set this to false to skip the time limit.

if timeLimit:
    expireDate = datetime.datetime(2017, 3, 28, 0, 0)
    today = datetime.datetime.now()   
    hasExpired = today > expireDate
    if hasExpired:
        print("Expire date: %04d-%02d-%02d"%(expireDate.year, expireDate.month, expireDate.day))
        print("Date today:  %04d-%02d-%02d"%(today.year, today.month, today.day))
        print("")
        print("The time trial has expired.")
        print("PiiGAB M-Bus Browse will exit.")
        print("Please contact PiiGAB to extend the trial time.")
        print("")
        sys.exit()

# =============================================================================
# === No args output ==========================================================
# =============================================================================
if len(sys.argv) < 5:
    #print("Usage: %s <IP-address> <IP-Port> <protocol> <M-Bus address> <inti*> <timeout*> <toggle*> <verbose*>"%(os.path.splitext(os.path.basename(sys.argv[0]))[0]))
    print("Usage: %s <IP> <port> <protocol> <M-Bus address> <option 1*> <option n*>"%(os.path.splitext(os.path.basename(sys.argv[0]))[0]))
    print("")
    print("Protocol:")
    print("\tu = UDP/IP.")
    print("\tt = TCP/IP.")
    print("")
    print("M-Bus address:")
    print("\t1-3 digits\t\t\tPrimary address.")
    print("\tXXXXXXXX\t\t\tIdentification.")
    print("\tXXXXXXXX.MMMM.VV.MM\t\tSecondary address.")
    print("\tXXXXXXXX:NNNNNNNN\t\tIdentification and node.")
    print("\tXXXXXXXX.MMMM.VV.MM:NNNNNNNN\tSecondary address and node.")
    print("")
    print("Init:\t\t\tOptional parameter: Init before REQ_UD2.")
    print("\t-i n\t\tNo init.")
    print("\t-i s\t\tSND_NKE.")
    print("\t-i a\t\tAPP_RST.")
    print("")
    print("Timeout:\t\tOptional parameter: Timeout.")
    print("\t-t 2000\t\t2000ms.")
    print("")
    print("Toggle:\t\t\tOptional parameter: Set control field for REQ_UD2.")
    print("\t-c 7B\t\tREQ_UD2 where C-field = 7B.")
    print("\t-c 5B\t\tREQ_UD2 where C-field = 5B.")
    print("")
    print("Debug:\t\t\tOptional parameter: Display debug.")
    print("\t-d sr\t\tDisplay sent and received data.")
    print("\t-d r\t\tDisplay record data.")
    print("\t-d a\t\tAll debug information.")
    print("")
    sys.exit()

# =============================================================================
# === Data types ==============================================================
# =============================================================================

class InitFlags(Enum):
    No = 0
    SND_NKE = 1
    APP_RST = 2

class DebugFlags(Enum):
    No = 0
    SendReceive = 1
    RecordData = 2
    All = 3

# Look-up dictionaries for arguments.
inits = {"n":InitFlags.No, "s":InitFlags.SND_NKE, "a":InitFlags.APP_RST }
debugs = { "n":DebugFlags.No, "sr":DebugFlags.SendReceive, "r":DebugFlags.RecordData, "a":DebugFlags.All }

# =============================================================================
# === Check args ==============================================================
# =============================================================================

# Try to get IP-address.
ip = ""
try:
    ipaddress.ip_address(sys.argv[1])
    ip = sys.argv[1]
except:
    print("Illegal IP-address")
    sys.exit()

# Try to get port.
port = 0
try:
    port = int(sys.argv[2])
except:
    print("Invalid port number")
    sys.exit()

# Try get protocol.
protocol = sys.argv[3]
if protocol != "u" and protocol != "t":
    print("Invalid protocol")
    sys.exit()

length = len(sys.argv)
adr = sys.argv[4]

parser = argparse.ArgumentParser()
parser.add_argument("-i", dest="Init", type=str, default="s")
parser.add_argument("-t", dest="Timeout", type=int, default=3000)
parser.add_argument("-c", dest="Toggle", type=str, default="7B")
parser.add_argument("-d", dest="Debug", type=str, default="n")
args = parser.parse_args(sys.argv[5:])

init = inits[args.Init]
timeout = args.Timeout / 1000
toggle = 0x7B if args.Toggle == "7B" else 0x5B if args.Toggle == "5B" else 0x7B
debug = debugs[args.Debug]

if debug == DebugFlags.All:
    print("Parameters (%s)"%(length-1))
    print("\tIP:\t\t%s"%(ip))
    print("\tPort:\t\t%s"%(port))
    print("\tProtocol:\t%s (%s)"%(protocol, "UDP" if protocol == "u" else "TCP"))
    print("\tAddress:\t%s"%(adr))
    print("\tInit:\t\t%s"%(init.name))
    print("\tTimeout:\t%ss"%(timeout))
    print("\tToggle:\t\t%02X"%(toggle))
    print("\tDebug:\t\t%s"%(debug.name))
    print("\t")
    print("")


# =============================================================================
# === Help tools for M-Bus decoding ===========================================
# =============================================================================

# List of device types.
DeviceTypes = {
    # All missing values are reserved.
    0x00:"Other", 0x01:"Oil", 0x02:"Electricity", 0x03:"Gas",
    0x04:"Heat", 0x05:"Steam", 0x06:"Warm water", 0x07:"Water",
    0x08:"Heat cost allocator", 0x09:"Compressed air", 0x0A:"Cooling (outlet)", 0x0B:"Cooling (inlet)",
    0x0C:"Heat (inlet)", 0x0D:"Heat/cooling", 0x0E:"Bus/System", 0x0F:"Unknown",
    0x14:"Calorific value", 0x15:"Hot water", 0x16:"Cold water", 0x17:"Dual water",
    0x18:"Pressure", 0x19:"A/D converter", 0x1A:"Smoke detector", 0x1B:"Room sensor",
    0x1C:"Gas detector", 0x20:"Breaker", 0x21:"Valve", 0x25:"Customer unit",
    0x28:"Waste water", 0x29:"Garbage", 0x31:"Gateway", 0x32:"Unidirectional repeater",
    0x33:"Bidirectional repeater", 0x36:"Radio converter (system)", 0x37:"Radio converter (meter)",
    0x38:"Bus converter"
}

# List of days in week
DaysInWeek = {
    0x00:"Not specified", 0x01:"Monday", 0x02:"Tuesday", 0x03:"Wednesday",
    0x04:"Thursday", 0x05:"Friday", 0x06:"Saturday", 0x07:"Sunday"
}

# Combinable (orthogonal) VIFE-codes.
def getCombinableVIFE(vife):
    medium = ""
    exponent = 0
    e = False
    
    if (vife & 0x70) == 0x00:
        medium = "See clause 9 and 8.4"
        exponent = (vife&0x0F)
        e = True
    elif (vife & 0x7F) == 0x14:
        medium = "relative deviation"
    elif ((vife & 0x7F) >= 0x15) and ((vife & 0x7F) <= 0x1C):
        medium = "record Error codes (slave to master), see 8.4"
    elif (vife & 0x7F) == 0x1D:
        medium = "standard conform data content"
    elif (vife & 0x7F) == 0x1E:
        medium = "compact profile with registers"
    elif (vife & 0x7F) == 0x1F:
        medium = "compact profile without registers"
    elif (vife & 0x7F ) == 0x20:
        medium = "per second"
    elif (vife & 0x7F ) == 0x21:
        medium = "per minute"
    elif (vife & 0x7F ) == 0x22:
        medium = "per hour"
    elif (vife & 0x7F ) == 0x23:
        medium = "per day"        
    elif (vife & 0x7F ) == 0x24:
        medium = "per week"
    elif (vife & 0x7F ) == 0x25:
        medium = "per month"    
    elif (vife & 0x7F ) == 0x26:
        medium = "per year"
    elif (vife & 0x7F ) == 0x27:
        medium = "per revolution / measurement"
    elif (vife & 0x7E) == 0x28:
        exponent = vife & 0x01
        medium = "increment per input pulse on input channel number %s"%(exponent)
    elif (vife & 0x7E) == 0x2A:
        exponent = vife & 0x01
        medium = "increment per output pulse on output channel number %s"%(exponent)
    elif (vife & 0x7F) == 0x2C:
        medium = "per l"
    elif (vife & 0x7F) == 0x2D:
        medium = "per m^3"
    elif (vife & 0x7F) == 0x2E:
        medium = "per kg"
    elif (vife & 0x7F) == 0x2F:
        medium = "per K"
    elif (vife & 0x7F) == 0x30:
        medium = "per kWh"
    elif (vife & 0x7F) == 0x31:
        medium = "per GJ"
    elif (vife & 0x7F) == 0x32:
        medium = "per kW"
    elif (vife & 0x7F) == 0x33:
        medium = "per K*l"
    elif (vife & 0x7F) == 0x34:
        medium = "per V"
    elif (vife & 0x7F) == 0x35:
        medium = "per A"
    elif (vife & 0x7F) == 0x36:
        medium = "multiplied by s"
    elif (vife & 0x7F) == 0x37:
        medium = "multiplied by s / V"
    elif (vife & 0x7F) == 0x38:
        medium = "multiplied by s / A"
    elif (vife & 0x7F) == 0x39:
        medium = "Start date(/time) of"
    elif (vife & 0x7F) == 0x3A:
        medium = "VIF contains uncorrected unit or value at metering conditions instead of converted unit"
    elif (vife & 0x7F) == 0x3B:
        medium = "Accumulation only if positive contributions"
    elif (vife & 0x7F) == 0x3C:
        medium = "Accumulation of abs value only if negative contribution"
    elif (vife & 0x7F) == 0x3D:
        medium = "Reserved"
    elif (vife & 0x7F) == 0x3E:
        medium = "value at base conditions"
    elif (vife & 0x7F) == 0x3F:
        medium = "OBIS-declaration"
    
        
    return medium

# Plain-text VIF.
def getPlainASCII(length, ascii):
    ascii = bytearray(ascii)
    ascii.reverse()
    return "".join(chr(i) for i in ascii)
    # Kolla Elvaco prim.adr 19.

# Third extension VIFE-codes.
def getVIFE_EF(vife):
    #print("VIFE = %02X"%(vife))
    # Reserved for future use. Not implemented in EN13757-3.
    return ""

# Second extension VIFE-codes.    
def getVIFE_FD(vife):
    #print("VIFE = %02X"%(vife))
    unit = "Not implemented VIFE FD"
    physicalQuantity = ""
    exponent = 0
    e = False
    
    if (vife & 0x7C) == 0x00:
        unit = "Currency units"
        physicalQuantity = "Credit, the nominal local legal currency units"
        exponent = (vife&0x03)-3
        e = True
    elif (vife & 0x7C) == 0x04:
        unit = ""
        physicalQuantity = "Debit, the nominal local legal currency units"
        exponent = (vife&0x03)-3
        e = True
    elif (vife & 0x7F) == 0x08:
        unit = ""
        physicalQuantity = "Unique message identification"
    elif (vife & 0x7F) == 0x09:
        unit = ""
        physicalQuantity = "Device type"
    elif (vife & 0x7F) == 0x0A:
        unit = ""
        physicalQuantity = "Manufacturer"
    elif (vife & 0x7F) == 0x0B:
        unit = "Enhanced identification"
        physicalQuantity = "Parameter set identification"
    elif (vife & 0x7F) == 0x0C:
        unit = ""
        physicalQuantity = "Model / Version"
    elif (vife & 0x7F) == 0x0D:
        unit = ""
        physicalQuantity = "Hardware version number"
    elif (vife & 0x7F) == 0x0E:
        unit = ""
        physicalQuantity = "Metrology (firmware) version number"
    elif (vife & 0x7F) == 0x0F:
        unit = ""
        physicalQuantity = "Other software version number"
    elif (vife & 0x7F) == 0x10:
        unit = ""
        physicalQuantity = "Customer location"
    elif (vife & 0x7F) == 0x11:
        unit = ""
        physicalQuantity = "Customer"
    elif (vife & 0x7F) == 0x12:
        unit = ""
        physicalQuantity = "Access code user"
    elif (vife & 0x7F) == 0x13:
        unit = "Improved selection"
        physicalQuantity = "Access code operator"
    elif (vife & 0x7F) == 0x14:
        unit = "and other user requirements"
        physicalQuantity = "Access code system operator"

    elif (vife & 0x7F) == 0x15:
        unit = ""
        physicalQuantity = "Access code developer"
    elif (vife & 0x7F) == 0x16:
        unit = ""
        physicalQuantity = "Password"
    elif (vife & 0x7F) == 0x17:
        unit = ""
        physicalQuantity = "Error flags"
    elif (vife & 0x7F) == 0x18:
        unit = ""
        physicalQuantity = "Error mask"
    elif (vife & 0x7F) == 0x19:
        unit = ""
        physicalQuantity = "Security key"
    elif (vife & 0x7F) == 0x1A:
        unit = ""
        physicalQuantity = "Digital output"
    elif (vife & 0x7F) == 0x1B:
        unit = ""
        physicalQuantity = "Digital input"
    elif (vife & 0x7F) == 0x1C:
        unit = ""
        physicalQuantity = "Baud rate [baud]"
    elif (vife & 0x7F) == 0x1D:
        unit = ""
        physicalQuantity = "Response delay time [bit-times]"
    elif (vife & 0x7F) == 0x1E:
        unit = ""
        physicalQuantity = "Retry"
    elif (vife & 0x7F) == 0x1F:
        unit = ""
        physicalQuantity = "Remote control"
    elif (vife & 0x7F) == 0x20:
        unit = ""
        physicalQuantity = "First storage number for cyclic storage"
    elif (vife & 0x7F) == 0x21:
        unit = ""
        physicalQuantity = "Last storage number for cyclic storage"
    elif (vife & 0x7F) == 0x22:
        unit = ""
        physicalQuantity = "Size of storage block"

    elif (vife & 0x7C) == 0x24:
        exponent = (vife&0x03)
        unit = ""
        unit = "second(s)" if exponent == 0x00 else unit
        unit = "minute(s)" if exponent == 0x01 else unit
        unit = "hour(s)" if exponent == 0x02 else unit
        unit = "days(s)" if exponent == 0x03 else unit
        physicalQuantity = "Storage interval [sec(s) ... day(s)]"
    elif (vife & 0x7F) == 0x28:
        unit = ""
        physicalQuantity = "Storage interval month(s)"
    elif (vife & 0x7F) == 0x29:
        unit = ""
        physicalQuantity = "Storage interval year(s)"
    elif (vife & 0x7F) == 0x2A:
        unit = ""
        physicalQuantity = "Operator specific data"
    elif (vife & 0x7F) == 0x2B:
        unit = ""
        physicalQuantity = "Time point second"
    elif (vife & 0x7C) == 0x2C:
        exponent = (vife&0x03)
        unit = ""
        unit = "second(s)" if exponent == 0x00 else unit
        unit = "minute(s)" if exponent == 0x01 else unit
        unit = "hour(s)" if exponent == 0x02 else unit
        unit = "days(s)" if exponent == 0x03 else unit
        physicalQuantity = "Duration since last readout [sec(s) ... day(s)]"
    elif (vife & 0x7F) == 0x30:
        unit = ""
        physicalQuantity = "Start (date/time) of tariff"
    elif (vife & 0x7C) == 0x30:
        unit = (vife&0x03)
        physicalQuantity = "Duration of tariff"
    elif (vife & 0x7C) == 0x34:
        exponent = (vife&0x03)
        unit = ""
        unit = "second(s)" if exponent == 0x00 else unit
        unit = "minute(s)" if exponent == 0x01 else unit
        unit = "hour(s)" if exponent == 0x02 else unit
        unit = "days(s)" if exponent == 0x03 else unit
        physicalQuantity = "Period of tariff [sec(s) ... day(s)]"
    elif (vife & 0x7F) == 0x38:
        unit = ""
        physicalQuantity = "Period of tariff month(s)"
    elif (vife & 0x7F) == 0x39:
        unit = ""
        physicalQuantity = "Period of tariff year(s)"
    elif (vife & 0x7F) == 0x3A:
        unit = ""
        physicalQuantity = "Dimensionless / no VIF"
    elif (vife & 0x7F) == 0x3B:
        unit = ""
        physicalQuantity = "Data container for wireless M-Bus protocol"
    elif (vife & 0x7C) == 0x3C:
        exponent = (vife&0x03)
        unit = ""
        unit = "second(s)" if exponent == 0x00 else unit
        unit = "minute(s)" if exponent == 0x01 else unit
        unit = "hour(s)" if exponent == 0x02 else unit
        unit = "days(s)" if exponent == 0x03 else unit
        physicalQuantity = "Period of nominal data transmission [sec(s) ... day(s)]"
    elif (vife & 0x70) == 0x40:
        unit = "V"
        physicalQuantity = "Volts"
        exponent = (vife&0x0F)-9
        e = True
    elif (vife & 0x70) == 0x50:
        unit = "A"
        physicalQuantity = "Current"
        exponent = (vife&0x0F)-12
        e = True        
    elif (vife & 0x7F) == 0x60:
        unit = ""
        physicalQuantity = "Reset counter"
    elif (vife & 0x7F) == 0x61:
        unit = ""
        physicalQuantity = "Cumulation counter"
    elif (vife & 0x7F) == 0x62:
        unit = ""
        physicalQuantity = "Control signal"
    elif (vife & 0x7F) == 0x63:
        unit = ""
        physicalQuantity = "Day of week"
    elif (vife & 0x7F) == 0x64:
        unit = ""
        physicalQuantity = "Week number"
    
    elif (vife & 0x7F) == 0x65:
        unit = ""
        physicalQuantity = "Time point of day change"
    elif (vife & 0x7F) == 0x66:
        unit = ""
        physicalQuantity = "State of parameter activation"
    elif (vife & 0x7F) == 0x67:
        unit = ""
        physicalQuantity = "Special supplier information"
    elif (vife & 0x7C) == 0x68:
        exponent = (vife&0x03)
        unit = ""
        unit = "hour(s)" if exponent == 0x00 else unit
        unit = "days(s)" if exponent == 0x01 else unit
        unit = "month(s)" if exponent == 0x02 else unit
        unit = "year(s)" if exponent == 0x03 else unit
        physicalQuantity = "Duration since last cumulation [hour(s) ... years(s)]"
    elif (vife & 0x7C) == 0x6C:
        exponent = (vife&0x03)
        unit = ""
        unit = "hour(s)" if exponent == 0x00 else unit
        unit = "days(s)" if exponent == 0x01 else unit
        unit = "month(s)" if exponent == 0x02 else unit
        unit = "year(s)" if exponent == 0x03 else unit
        physicalQuantity = "Operating time battery [hour(s) ... years(s)]"
    elif (vife & 0x7F) == 0x70:
        unit = ""
        physicalQuantity = "Date and time of battery charge"
    elif (vife & 0x7F) == 0x71:
        unit = "dBm"
        physicalQuantity = "RF level units"
    elif (vife & 0x7F) == 0x72:
        unit = ""
        physicalQuantity = "Daylight savings"
    elif (vife & 0x7F) == 0x73:
        unit = ""
        physicalQuantity = "Listening window management"
    elif (vife & 0x7F) == 0x74:
        unit = "days"
        physicalQuantity = "Remaining battery life time"
    elif (vife & 0x7F) == 0x75:
        unit = ""
        physicalQuantity = "Number of time the meter was stopped"
    elif (vife & 0x7F) == 0x76:
        unit = ""
        physicalQuantity = "Data container for manufacturer specific protocol"
                    
    ret = ""
    if e:
        ret = "{:s} {:4s} E{:d}".format(physicalQuantity, unit, exponent)
    else:
        ret = "{:s} {:4s}".format(physicalQuantity, unit)
    return ret 

# First extension VIFE-codes.    
def getVIFE_FB(vife):
    #print("VIFE = %02X"%(vife))
    unit = "Not implemented VIFE FB"
    physicalQuantity = ""
    exponent = 0
    e = False
    
    if (vife & 0x7E) == 0x00:
        unit = "MWh"
        physicalQuantity = "Energy"
        exponent = (vife&0x01)-1
        e = True
    elif (vife & 0x7E) == 0x02:
        unit = "kVARh"
        physicalQuantity = "Reactive energy"
        exponent = (vife&0x01)
        e = True
    elif (vife & 0x7E) == 0x04:
        unit = "kVAh"
        physicalQuantity = "Apparent energy"
        exponent = (vife&0x01)
        e = True
    elif (vife & 0x7E) == 0x08:
        unit = "GJ"
        physicalQuantity = "Energy"
        exponent = (vife&0x01)-1
        e = True
    elif (vife & 0x7C) == 0x0C:
        unit = "MCal"
        physicalQuantity = "Energy"
        exponent = (vife&0x03)-1
        e = True
    elif (vife & 0x7E) == 0x10:
        unit = "m^3"
        physicalQuantity = "Volume"
        exponent = (vife&0x01)+2       # Be GIG ACK
        e = True
    elif (vife & 0x7C) == 0x14:
        unit = "kVAR"
        physicalQuantity = "Reactive power"
        exponent = (vife&0x03)-3
        e = True
    elif (vife & 0x7E) == 0x18:
        unit = "t"
        physicalQuantity = "Mass"
        exponent = (vife&0x01)+2       # Be GIG ACK
        e = True
    elif (vife & 0x7E) == 0x1A:
        unit = "%"
        physicalQuantity = "Relative humidity"
        exponent = (vife&0x01)-1
        e = True
    elif (vife & 0x7F) == 0x20:
        unit = "feet^3"
        physicalQuantity = "Volume"
    elif (vife & 0x7F) == 0x21:
        unit = "0,1 feet^3"         # Be GIG ACK
        physicalQuantity = "Volume"
    elif (vife & 0x7E) == 0x28:
        unit = "MW"
        physicalQuantity = "Power"
        exponent = (vife&0x01)-1
        e = True
    elif (vife & 0x7F) == 0x2A:
        unit = "°"
        physicalQuantity = "Phase U-U (volt. to volt.)"
    elif (vife & 0x7F) == 0x2B:
        unit = "°"
        physicalQuantity = "Phase U-I (volt. to current.)"
    elif (vife & 0x7C) == 0x2C:
        unit = "Hz"
        physicalQuantity = "Frequency"
        exponent = (vife&0x03)-3
        e = True
    elif (vife & 0x7E) == 0x30:
        unit = "GJ/h"
        physicalQuantity = "Power"
        exponent = (vife&0x01)-1
        e = True
    elif (vife & 0x7C) == 0x34:
        unit = "kVA"
        physicalQuantity = "Apparent power"
        exponent = (vife&0x03)-3
        e = True
    elif (vife & 0x7C) == 0x74:
        unit = "°C"
        physicalQuantity = "Cold/warm temperature limit"
        exponent = (vife&0x03)-3
        e = True
    elif (vife & 0x78) == 0x78:    # Be GIG ACK
        unit = "W"
        physicalQuantity = "Cum. max. of active power"
        exponent = (vife&0x07)-3
        e = True
    else:
        unit = ""
        physicalQuantity = "Reserved"
      
    ret = ""
    if e:
        ret = "{:s} {:4s} E{:d}".format(physicalQuantity, unit, exponent)
    else:
        ret = "{:s} {:4s}".format(physicalQuantity, unit)
    return ret 

# Primary VIF-codes.
def getVIF(vif):
    #print("VIF = %02X"%(vif))
    unit = "Not implemented VIF"
    physicalQuantity = ""
    exponent=0
    e = False
    isDateTime = False

    if (vif & 0x78) == 0x00:
        unit = "Wh"
        physicalQuantity = "Energy"
        exponent = (vif&0x07)-3
        e = True;
    elif (vif & 0x78) == 0x08:
        unit = "J"
        physicalQuantity = "Energy"
        exponent = (vif&0x07)
        e = True;
    elif (vif & 0x78) == 0x10:
        unit = "m^3"
        physicalQuantity = "Volume"
        exponent = (vif&0x07)-6
        e = True;
    elif (vif & 0x78) == 0x18:
        unit = "kg"
        physicalQuantity = "Mass"
        exponent = (vif&0x07)-3
        e = True
    elif (vif & 0x7C) == 0x20:
        exponent = (vif&0x03)
        unit = ""
        unit = "second(s)" if exponent == 0x00 else unit
        unit = "minute(s)" if exponent == 0x01 else unit
        unit = "hour(s)" if exponent == 0x02 else unit
        unit = "days(s)" if exponent == 0x03 else unit
        physicalQuantity = "On time"
    elif (vif & 0x7C) == 0x24:
        exponent = (vif&0x03)
        unit = ""
        unit = "second(s)" if exponent == 0x00 else unit
        unit = "minute(s)" if exponent == 0x01 else unit
        unit = "hour(s)" if exponent == 0x02 else unit
        unit = "day(s)" if exponent == 0x03 else unit
        physicalQuantity = "Operating time"
    elif (vif & 0x78) == 0x28:
        unit = "W"
        physicalQuantity = "Power"
        exponent = (vif&0x07)-3
        e = True;
    elif (vif & 0x78) == 0x30:
        unit = "J/h"
        physicalQuantity = "Power"
        exponent = (vif&0x07)
        e = True;
    elif (vif & 0x78) == 0x38:
        unit = "m^3/h"
        physicalQuantity = "Volume flow"
        exponent = (vif&0x07)-6
        e = True;
    elif (vif & 0x78) == 0x40:
        unit = "m^3/min"
        physicalQuantity = "Volume flow ext"
        exponent = (vif&0x07)-7
        e = True;
    elif (vif & 0x78) == 0x48:
        unit = "m^3/s"
        physicalQuantity = "Volume flow ext"
        exponent = (vif&0x07)-9
        e = True;
    elif (vif & 0x78) == 0x50:
        unit = "kg/h"
        physicalQuantity = "Mass flow"
        exponent = (vif&0x07)-3
        e = True;
    elif (vif & 0x7C) == 0x58:
        unit = "°C"
        physicalQuantity = "Flow temperature"
        exponent = (vif&0x03)-3
        e = True;
    elif (vif & 0x7C) == 0x5C:
        unit = "°C"
        physicalQuantity = "Return temperature"
        exponent = (vif&0x03)-3
        e = True;
    elif (vif & 0x7C) == 0x60:
        unit = "K"
        physicalQuantity = "Temperature difference"
        exponent = (vif&0x03)-3
        e = True;
    elif (vif & 0x7C) == 0x64:
        unit = "°C"
        physicalQuantity = "External temperature"
        exponent = (vif&0x03)-3
        e = True;
    elif (vif & 0x7C) == 0x68:
        unit = "bar"
        physicalQuantity = "Pressure"
        exponent = (vif&0x03)-3
        e = True;
    elif (vif & 0x7F) == 0x6C:
        unit = "(depending on data field)"
        physicalQuantity = "Date"
        isDateTime = True
        e = False;
    elif (vif & 0x7F) == 0x6D:
        unit = "(depending on data field)"
        physicalQuantity = "Date/time"
        isDateTime = True
        e = False;
    elif (vif & 0x7F) == 0x6E:
        unit = "(dimensionless)"
        physicalQuantity = "Units for H.C.A."
        e = False;
    elif (vif & 0x7C) == 0x70:
        exponent = (vif&0x03)
        unit = ""
        unit = "second(s)" if exponent == 0x00 else unit
        unit = "minute(s)" if exponent == 0x01 else unit
        unit = "hour(s)" if exponent == 0x02 else unit
        unit = "day(s)" if exponent == 0x03 else unit
        physicalQuantity = "Averaging duration"
    elif (vif & 0x7C) == 0x74:
        exponent = (vif&0x03)
        unit = ""
        unit = "second(s)" if exponent == 0x00 else unit
        unit = "minute(s)" if exponent == 0x01 else unit
        unit = "hour(s)" if exponent == 0x02 else unit
        unit = "days(s)" if exponent == 0x03 else unit
        physicalQuantity = "Actuality duration"
    elif vif == 0x78:
        unit = ""
        physicalQuantity = "Fabrication no"
        e = False;
    elif vif == 0x79:
        unit = ""
        physicalQuantity = "(Enhanced) identification"
        e = False;
    elif vif == 0x7A:
        unit = ""
        physicalQuantity = "Address"
        e = False;          
    
    ret = ""
    if e:
        ret = "{:s} {:4s} E{:d}".format(physicalQuantity, unit, exponent)
    else:
        ret = "{:s} {:4s}".format(physicalQuantity, unit)
    return ret, isDateTime

# =============================================================================
# === Communication ===========================================================
# =============================================================================

def SetupSocket(protocol, ip, port, timeout):
    if protocol == "u":
        sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        sock.settimeout(timeout)
        return sock
    elif protocol == "t":
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.settimeout(timeout)
        sock.connect((ip, port))
        return sock
    else:
        print("Invalid protocol")
        sys.exit()

def SendAndReceive(request, sock, protocol, bytesToRead, ip = "", port = 0):
    if protocol == "u":
        sock.sendto(request, (ip, port))
        response = sock.recv(bytesToRead)
        return response
    elif protocol == "t":
        sock.send(request)
        response = sock.recv(bytesToRead)
        return response

# =============================================================================
# === Misc ====================================================================
# =============================================================================
    
def printhex(data):
    tstr = ""
    for j in range(len(data)):
        if j%16 == 0 and j != 0:
            tstr = tstr + "\n"
        tstr = tstr + "%02X "%data[j]
    print(tstr)
    print("")

# Print data as hex and include an offset before each line.    
def PrintHexWithOffset(data, offset):
    tstr = offset
    for j in range(len(data)):
        if j%16 == 0 and j != 0:
            tstr = tstr + "\n" + offset
        tstr = tstr + "%02X "%data[j]
    print(tstr)

# Print data as hex on a line.
def DataLineAsHex(data):
    line = "".join("%02X "%(i) for i in data)
    return line
    

def calcbcc(array):
    cc=0
    for i in range(len(array)):
        cc = cc+array[i]
    return cc&0xFF

def bcdToValue(data):
    try:
        data = "".join(chr(i) for i in data)
        return data
    except:
        return "Invalid BCD value. Enable debugging."
    
def bcdtoval(arr):
    # Kolla: Prim.Adr 4, objekt 35 innehåller 0xFF. Hur ska det tolkas?
    retval=0
    error = False
    mul=1
    for i in arr:
        q = i & 0x0F
        if q > 9:
            error = True
        retval += q*mul
        mul *= 10;
        q = (i & 0xF0) >> 4
        if q > 9:
            error = True
        retval += q*mul
        mul *= 10;
        
    return retval if error == False else "Invalid BCD value"

def SecondaryAddress(data):
    data = bytearray(data)
    id = data[0:4]
    id.reverse()
    id = "".join("%02X" % b for b in id)
    manu = data[4:6]
    manu.reverse()
    manu = "".join("%02X" % b for b in manu)
    ver = data[6]
    med = data[7]
    tmp = struct.unpack("<H",data[4:6])[0]
    name = "%c"%(((tmp>>10)&0x1F)+64)+"%c"%(((tmp>>5)&0x1F)+64)+"%c"%((tmp&0x1F)+64)
    type = DeviceTypes[med] if med in DeviceTypes else "Reserved"
    return id, manu, "%02X"%(ver), "%02X"%(med), name, type
    
def Status(value):
    result = ""
    
    # Status for bit 0 and 1.
    if (value & 0x03):
        bit1_2 = value & 0x03
        result = "Application busy" if bit1_2 == 1 else "Any application error" if bit1_2 == 2 else "Abnormal condition / alarm"
    
    # Status for bit 2.
    if (value & 0x04) == 0x04:
        result += ", " if len(result) > 0 else result
        result += "Power low"
        
    # Status for bit 3.
    if (value & 0x08) == 0x08:
        result += ", " if len(result) > 0 else result
        result += "Permanent error"
        
    # Status for bit 4.
    if (value & 0x10) == 0x010:
        result += ", " if len(result) > 0 else result
        result += "Temporary error"

    # Status for bit 5.
    if (value & 0x20) == 0x20:
        result += ", " if len(result) > 0 else result
        result += "Manufacturer bit1 set"

    # Status for bit 6.
    if (value & 0x40) == 0x40:
        result += ", " if len(result) > 0 else result
        result += "Manufacturer bit1 set"

    # Status for bit 7.
    if (value & 0x80) == 0x80:
        result += ", " if len(result) > 0 else result
        result += "Manufacturer bit1 set"
        
    return result

# Decode CP16 from INT16 (Type G) to YY-MM-DD.
def DecodeCP16(value):
    # Prim.adr 42 objekt 4. Prim.adr 43 objekt 4, 9, 11.
    y1 = (value & 0xF000) >> 8
    y2 = value & 0x00E0
    mon = (value & 0x0F00) >> 8
    day = value & 0x001F
    year = (y1 >> 1) | (y2 >> 5)
    return "%02d-%02d-%02d"%(year, mon, day)

# Decode CP24 from INT24 (Type J) to HH:MM:SS.
def DecodeCP24(value):
    sec = value & 0x00003F
    min = (value & 0x003F00) >> 8
    hour = (value & 0x1F0000) >> 16
    return "%02d:%02d:%02d"%(hour, min, sec)

# Decode CP32 from INT32 (Type F) to YY-MM-DD HH:MM.
def DecodeCP32(value):
    # Prim.adr 25, objekt 20.
    min  =  value & 0x0000003F
    hour = (value & 0x00001F00) >> 8
    day  = (value & 0x001F0000) >> 16
    mon  = (value & 0x0F000000) >> 24
    y1   = (value & 0xF0000000) >> 24
    y2   = (value & 0x00E00000) >> 16
    hy   = (value & 0x00006000) >> 8 # Hundred years
    year = (y1 >> 1) | (y2 >> 5)
    return "%02d-%02d-%02d %02d:%02d"%(year, mon, day, hour, min)

# Decode CP48 from INT48 (Type I) to YYYY-MM-DD HH:MM:SS, day, week, period.
def DecodeCP48(value):
    
    #     48    41   40    33   32    25   24    17   16     9   8      1
    #     47    40   39    32   31    24   23    16   15     8   7      0
    #DT = 00000000 | 00000000 | 00000000 | 00000000 | 00000000 | 00000000
    #     --______   ----____   ---_____   ---_____   -_------   _-______
    #     ds Week	 y1 mon     y2 day    dow hour   ih  min    lt  sec
    
    ds 	    = (value & 0xC00000000000) >> 40   # Daylight saving deviation (hour).
    week    = (value & 0x3F0000000000) >> 40
    y1	    = (value & 0x00F000000000) >> 32
    mon	    = (value & 0x000F00000000) >> 32
    y2	    = (value & 0x0000E0000000) >> 24
    day	    = (value & 0x00001F000000) >> 24
    dow	    = (value & 0x000000E00000) >> 21   # Day of week.
    hour	= (value & 0x0000001F0000) >> 16
    i	    = (value & 0x000000008000) >> 8    # Time invalid.
    h	    = (value & 0x000000004000) >> 8    # Daylight saving deviation (hour).
    min	    = (value & 0x000000003F00) >> 8
    l	    = (value & 0x000000000080)         # Leap year.
    per     = (value & 0x000000000040) >> 6    # Time during daylight savings, period.
    sec	    = (value & 0x00000000003F)
    
    year = 2000 + (y1 >> 1) + (y2 >> 5)
    dayInWeek = DaysInWeek[dow]                # Day in week.
    period = "winter" if per == 0x00 else "summer"
    
    string =  "%04d-%02d-%02d %02d:%02d:%02d %s V%02d %s"%(year, mon, day, hour, min, sec, dayInWeek, week, period)
    return string

# Decode a BCD to DateTime. Unsupported coding of date and time, but is common for ABB and BMeters.
def DecodeBCD12(value):
    dateTime = "%012d"%(value)
    year = dateTime[0:2]
    mon =  dateTime[2:4]
    day =  dateTime[4:6]
    hour = dateTime[6:8]
    min =  dateTime[8:10]
    sec =  dateTime[10:12]
        
    return "%s-%s-%s %s:%s:%s"%(year, mon, day, hour, min, sec)
    
# Decode CP16, CP24, CP32 and CP48.
def DecodeDateTime(dataType, value):
    # INT16 = CP16.
    if dataType == 0x02:
        vib = "Date YY-MM-DD"
        return DecodeCP16(value), vib
    
    # INT24 = CP24.
    elif dataType == 0x03:
        vib = "Time HH:MM:SS"
        return DecodeCP24(value), vib
        
    # INT32 = CP32.
    elif dataType == 0x04:
        vib = "Date & time YY-MM-DD HH:MM"
        return DecodeCP32(value), vib
        
    # INT48 = CP48.
    elif dataType == 0x06:
        vib = "Date & time YYYY-MM-DD HH:MM:SS, day, week, period"
        return DecodeCP48(value), vib
        
    # BCD12 = Unsupported date and time.
    elif dataType == 0x0E:
        vib = "Date & time YY-MM-DD HH:MM:SS (Unsupported in EN13757)"
        return DecodeBCD12(value), vib

# =============================================================================        
# === Decode object ===========================================================
# =============================================================================

def decode_field(arr):
    dif=arr[0]      # Get DIF.
    #print("\tDIF\t\t= 0x%02X"%dif)
    
    value=None
    dataTypeStr=""              # String representation of the data type.
    recordLength=1              # Point to second byte in arr, may be DIFE or VIF.
    more_telegrams=False
    mfct_data=False
    functionFields=["Instantaneous value", "Maximum value", "Minimum value", "Value during error state"]
    function=functionFields[(dif>>4)&0x03]
    dataType = dif & 0x0F       # Get data type.
    vib = ""
    isDateTime = False
    hasManufacturerSpecificVIB = False
    
    # === Parse DIB ===========================================================
          
    # DIF = MDH.
    if dataType == 0x0F:
        mdh = (dif>>4)&0x07  # Get MDH.
        dataTypeStr = "0x%sF"%(mdh)
        
        # MDH = 0x0F.
        if mdh==0:
            function="Start of manufacturer specific data structures to end of user data"
            mfct_data=True
            vib = "Start of manufacturer specific data"
            
        # MDH = 0x1F.
        elif mdh==1:
            function="Start of manufacturer specific data structures to end of user data, more messages"
            mfct_data=True
            more_telegrams=True
            vib = "Start of manufacturer specific data"
            
        # MDH = 0x2F.
        elif mdh==2:
            function="Idle filler, following byte is a DIF"
            vib = "Idle filler"
            
        # MDH = 0x7F.
        elif mdh==7:
            function="Global readout request"
            vib = "Global readout request"
        else:
            function="Reserved"
            vib = "Reserved"
            
        return (value, dataTypeStr, recordLength, function, mfct_data, more_telegrams, vib)
    
    # If DIFE is present, check all DIFE.
    elif dif & 0x80:
        for i in range(10):
            dife=arr[recordLength]              # Get DIFE.
            #print("\tDIFE\t\t= 0x%02X"%dife)
            recordLength = recordLength + 1     # Point to next byte.
            
            # Check if extension bit is set in current DIFE.
            if (dife&0x80)==0:
                break
    
    # === Parse VIB ===========================================================
    
    # Get VIB
    vif=arr[recordLength]  
    
    # Primary VIF-codes.
    if vif < 0xFB and vif != 0x7F: 
        vib, isDateTime = getVIF(vif)
        
    # First extension of VIF-codes.
    elif vif == 0xFB:                   
        vife = arr[recordLength + 1]
        vib = getVIFE_FB(vife)
        
    # Plain ASCII.
    elif vif == 0x7C or vif == 0xFC:   
        # Note:
        # The VIFE will follow after the length and data.
        # Ex1: VIF LEN ASCII_1 ... ASCII_N VIFE DATA    (VIFE present).
        # Ex2: VIF LEN ASCII_1 ... ASCII_N DATA         (No VIFE present).
        length = arr[recordLength + 1]
        # Point first byte after length and include as many bytes as specified in length.
        ascii = arr[recordLength+2:recordLength+2+arr[recordLength+1]]
        vib = getPlainASCII(length, ascii)
        recordLength += length + 1 
    
    # Second extension of VIF-codes.
    elif vif == 0xFD:
        vife = arr[recordLength + 1]
        vib = getVIFE_FD(vife)
        
    # (Reserved) Third extension VIF-codes.
    elif vif == 0xEF:
        vife = arr[recordLength + 1]
        vib = getVIFE_EF(vife)
        
    # Any VIF.
    elif vif == 0x7E or vif == 0xFE:
        vib = "Any VIF"
        
    # Manufacturer specific.
    elif vif == 0x7F or vif == 0xFF:
        vib = "Manufacturer specific"
        hasManufacturerSpecificVIB = True
        
    # Unknown VIF.
    else:
        vib = "Unknown"   
    
    recordLength = recordLength + 1 # Point to next byte (VIFE or start of data area).
    
    #print("VIF = 0x%02X"%(vif))
    
    # If VIFE is present, check all VIFE.
    if vif & 0x80:
        for i in range(10):
            vife=arr[recordLength]              # Get VIFE.
            #print("VIFE = 0x%02X"%vife)
            
            #if hasManufacturerSpecificVIB == False:
            #    comb = getCombinableVIFE(arr[recordLength+1])  # Combinable (orthogonal) VIFE-codes.
            #    print(comb)
            #    vib += comb
            
            recordLength = recordLength + 1     # Point to next byte.
            
            # Check if extension bit is set in current VIFE.
            if (vife&0x80)==0:
                break

    # === Parse DATA ==========================================================
                
    # No data.
    if dataType==0:
        dataTypeStr="No Data"
        
    # INT8.
    elif dataType==0x01:
        dataTypeStr="INT8"
        value=struct.unpack('<b',arr[recordLength:recordLength+1])
        value = value[0]
        recordLength = recordLength + 1
        
    # INT16.
    elif dataType==0x02:
        dataTypeStr="INT16"
        value=struct.unpack('<h',arr[recordLength:recordLength+2])
        value = value[0]
        recordLength = recordLength + 2
        
    # INT24.
    elif dataType==0x03:
        dataTypeStr="INT24"
        value = struct.unpack('<i', b'\x00' + arr[recordLength:recordLength + 3])
        value = value[0] >> 8
        recordLength = recordLength + 3
        
    # INT32.
    elif dataType==0x04:
        dataTypeStr="INT32"
        value=struct.unpack('<i',arr[recordLength:recordLength+4])
        value = value[0]
        recordLength = recordLength + 4
        
    # FLOAT.
    elif dataType==0x05:
        dataTypeStr="REAL32"
        value=struct.unpack('<f',arr[recordLength:recordLength+4])
        value = value[0]
        recordLength = recordLength + 4
        
    # INT48.
    elif dataType==0x06:
        dataTypeStr="INT48"
        value=struct.unpack('<q',b'\x00\x00' + arr[recordLength:recordLength+6])
        value = value[0] >> 16
        recordLength = recordLength + 6
        
    # INT64.
    elif dataType==0x07:
        dataTypeStr="INT64"
        value=struct.unpack('<q',arr[recordLength:recordLength+8])
        value = value[0]
        recordLength = recordLength + 8
        
    # Selection for readout.
    elif dataType==0x08:
        dataTypeStr="SEL"
        recordLength = recordLength + 8
        
    # BCD2.
    elif dataType==0x09:
        dataTypeStr="BCD2"
        value=bcdtoval(arr[recordLength:recordLength+1])
        recordLength = recordLength + 1
        
    # BCD4.
    elif dataType==0x0A:
        dataTypeStr="BCD4"
        value=bcdtoval(arr[recordLength:recordLength+2])
        recordLength = recordLength + 2
        
    # BCD6.
    elif dataType==0x0B:
        dataTypeStr="BCD6"
        value=bcdtoval(arr[recordLength:recordLength+3])
        recordLength = recordLength + 3
        
    # BCD8.
    elif dataType==0x0C:
        dataTypeStr="BCD8"
        value=bcdtoval(arr[recordLength:recordLength+4])
        recordLength = recordLength + 4
        
    # Variable length.
    elif dataType==0x0D:
        LVAR=arr[recordLength]
        dataLength = 0
        recordLength = recordLength + 1
        dataTypeStr = "LVAR"
        
        # 8-bit text (ISO 8859-1).
        # There is a bug in Browse. The LVAR container for wireless is not shown correctly.
        if arr[recordLength-3] == 0xFD and arr[recordLength-2] == 0x3B: # Wireless container.
            dataLength = LVAR
            data = arr[recordLength:recordLength+dataLength]
            value = "".join("%02X "%(i) for i in data[::-1])            
        elif LVAR <= 0xBF:
            dataLength = LVAR
            data = arr[recordLength:recordLength+dataLength]
            #value = "".join("%02X "%(i) for i in data[::-1])   # Byte ouput.
            value = "".join(chr(i) for i in data[::-1])         # Char ouput.
            
        # Positive BCD number.
        elif LVAR >= 0xC0 and LVAR <= 0xCF:
            dataLength = LVAR & 0x0F
            value=b""
            for i in range(dataLength):
                value += b"%02x"%arr[recordLength+dataLength-i-1]
            value = value.hex().upper()

        # Negative BCD number.            
        elif LVAR >= 0xD0 and LVAR <= 0xDF:
            dataLength = LVAR & 0x0F
            value=b"-"
            for i in range(dataLength):
                value += b"%02x"%arr[recordLength+dataLength-i-1]
            value = value.hex().upper()
            
        # Binary number with (LVAR - 0xE0) bytes.
        elif LVAR >= 0xE0 and LVAR <= 0xEF:
            dataLength = LVAR & 0x0F
            value=b""
            for i in range(dataLength):
                value += b"%02x"%arr[recordLength+i]
            value = value.hex().upper()
            
        # Binary number with 4*(LVAR - 0xEC) bytes.
        elif LVAR >= 0xF0 and LVAR <= 0xF4:
            dataLength = LVAR & 0x0F
            value=b""
            for i in range(dataLength):
                value += b"%02x"%arr[recordLength+i]
            value = value.hex().upper()
            
        # Binary number with 48 bytes.
        elif LVAR == 0xF5:
            dataLength = 48
            value = "Binary 48 byte value"
            
        # Binary number with 68 bytes.
        elif LVAR == 0xF6:
            dataLenght = 64
            value = "Binary 68 byte value"
            
        # Reserved
        else:
            dataLenght = 0
            value = "Reserved"
        
        recordLength = recordLength + dataLength
        
    # BCD12.
    elif dataType==0x0E:
        dataTypeStr="BCD12"
        value=bcdtoval(arr[recordLength:recordLength+6])
        recordLength = recordLength + 6
        
    # Unknown data type.
    else:
        dataTypeStr="N/A"
    
    # Decode value to date or time.
    if isDateTime and (dataType == 0x02 or dataType == 0x03 or dataType == 0x04 or dataType == 0x06 or dataType == 0x0E):
        value, vib = DecodeDateTime(dataType, value)
    
    return (value, dataTypeStr, recordLength, function, mfct_data, more_telegrams, vib)
    
# =============================================================================
# === Help functions ==========================================================
# =============================================================================

# Select the records that should be saved to a template.
def SelectRecordsForTemplate(records):
    print("Type in data record numbers, example [1 2 5 10]: ", end="")
    inp = input()
    
    # Exit if no records is given.
    if inp == "":
        return
        
    numbers = inp.split()
    list = {}
    
    for number in numbers:
        number = int(number)
        if number in records:
            list[number] = records[number]
    
    return list

# Setup the folder to store the template in.
def SetupTemplateFolder(folderPath):
    if os.path.isdir(folderPath) == False:
        os.makedirs(folderPath)

# Create the file path to the template file.
def CreateTemplateFile(templateFolderPath, manufacturer, version, media):
    fileName = "%s_%s_%s"%(manufacturer, version, media)
    filePath = "%s%s%s.tpl"%(templateFolderPath, os.sep, fileName)
    return filePath

# Creates the template data from selected records.
def CreateTemplateData(records, telegramCounter, primaryAddress, secondaryAddress):
    list = []
    
    deviceName = "%s_%s_%s"%(secondaryAddress[1], secondaryAddress[2], secondaryAddress[3])
    address = "FFFFFFFF.%s.%s.%s"%(secondaryAddress[1], secondaryAddress[2], secondaryAddress[3])
    device = "[Device],,,%s,,,,,,,,,,%s,,,,,,%s,,,,,,\n"%(deviceName, telegramCounter, address)
    
    list.append(device)
    
    for record in records:
        tagName = records[record][6]
        dataType = records[record][1]
        tag = "[Tag],,,%s,%s,,,,,,,,,,,,,,,%s,1,%s1,,,,,,,,,,,,,,,,,,,,,,,\n"%(deviceName,tagName,record,dataType)
        list.append(tag)
        
    return list

# Create the template.
def CreateTemplate(data, templateFilePath):
    file = open(templateFilePath, 'w')
    for line in data:
        file.write(line)    # Write row to file
    file.close()

def CreateSLV_SEL(identification, manufacturer, version, media, node):
    # SLV_SEL without enhanced address: 68 0B 0B 68 C ADR CI XX XX XX XX MM MM VV MM CS 16.
    # SLV_SEL with enhanced address:    68 11 11 68 C ADR CI XX XX XX XX MM MM VV MM 0C 78 NN NN NN NN CS 16.
    
    length = 0x0B if node == None else 0x11 # Get length based on enhanced seconday address should be used.
    
    # Create SLV_SEL command.    
    slv_sel = b""
    slv_sel += struct.pack("BBBB", 0x68, length, length, 0x68)  # Add frame charecters to SLV_SEL.
    slv_sel += struct.pack("BBB", 0x53, 0xFD, 0x52)             # Add control and address fields to SLV_SEL.
    slv_sel += struct.pack("<I",identification)                 # Add identification to SLV_SEL.
    slv_sel += struct.pack("<H", manufacturer)                  # Add manufacturer to SLV_SEL.
    slv_sel += struct.pack("B", version)                        # Add verison to SLV_SEL.
    slv_sel += struct.pack("B", media)                          # Add media to SLV_SEL.
    
    # Check if the enhanced address should be used.
    if node != None:
        slv_sel += struct.pack("BB", 0x0C, 0x78)    # Add frame charecters to SLV_SEL.
        slv_sel += struct.pack("<I", node)          # Add node address to SLV_SEL.
    
    cs = calcbcc(slv_sel[4:])               # Calculated checksum.
    slv_sel += struct.pack("BB", cs, 0x16) # Add checksum and end to SLV_SEL.
    return slv_sel
    
# =============================================================================
# === Main ====================================================================
# =============================================================================

sock = SetupSocket(protocol, ip, port, timeout)
slv_sel = None
response=""

# Use primary address.
if len(adr) < 4: # Primary address.
    pass

# Use identifiction and wild cards as secondary address.
elif len(adr) == 8:
    # adr = XXXXXXXX
    id = int(adr[0:8], 16)
    mfct = 0xFFFF
    ver = 0xFF
    med = 0xFF
    node = None
    slv_sel = CreateSLV_SEL(id, mfct, ver, med, node)
    adr = "253" 

# Use identifiction and wild cards as secondary address with enhanced secondary address.
elif len(adr) == 17 and adr[8] == ":":
    # adr = XXXXXXXX:NNNNNNNN
    id = int(adr[0:8], 16)
    mfct = 0xFFFF
    ver = 0xFF
    med = 0xFF
    node = int(adr[9:], 16)
    slv_sel = CreateSLV_SEL(id, mfct, ver, med, node)
    adr = "253" 
    
# Use secondary address without enhanced as secondary address.
elif len(adr) == 19 and "." in adr and len(adr.split(".")) == 4:
    # adr = XXXXXXXX.MMMM.VV.MM
    parts = adr.split(".")
    id = int(parts[0], 16)
    mfct = int(parts[1], 16)
    ver = int(parts[2], 16)
    med = int(parts[3], 16)
    node = None
    
    slv_sel = CreateSLV_SEL(id, mfct, ver, med, node)
    adr = "253" 
    
# Use secondary address with enhanced as secondary address.
elif len(adr) == 28 and "." in adr and ":" in adr:
    # adr = XXXXXXXX.MMMM.VV.MM:NNNNNNNN
    sec = adr[:adr.index(":")]
    parts = sec.split(".")
    id = int(parts[0], 16)
    mfct = int(parts[1], 16)
    ver = int(parts[2], 16)
    med = int(parts[3], 16)
    
    node = adr[adr.index(":")+1:]
    node = int(node, 16)
    
    slv_sel = CreateSLV_SEL(id, mfct, ver, med, node)
    adr = "253"

# Select M-Bus meter with SLV_SEL.
if slv_sel != None:
    
    if debug == DebugFlags.SendReceive or debug == DebugFlags.All:
        print("SLV_SEL")
        printhex(slv_sel)
    
    response = None
    try:
        response = SendAndReceive(slv_sel, sock, protocol, 1, ip, port)
    except:
        print("Failed to receive data for SLV_SEL")
        sock.close()
        sys.exit()
    
    if debug == DebugFlags.SendReceive or debug == DebugFlags.All:
        print("ACK")
        printhex(response)
    
    # Check for none ACK response.
    if len(response) != 1 or response[0] != 0xE5:
        print("Invalid response for SLV_SEL")
        sys.exit()

# Send SND_NKE and receive ACK.
if init == InitFlags.SND_NKE:
    # Request format: 10 40 ADR CS 16.
    start = b"\x10\x40"+struct.pack("B",int(adr))
    end = struct.pack("B",calcbcc(start[1:])) + b"\x16"
    snd_nke = start + end
    
    if debug == DebugFlags.SendReceive or debug == DebugFlags.All:
        print("SND_NKE")
        printhex(snd_nke)
    
    response = None
    try:
        response = SendAndReceive(snd_nke, sock, protocol, 1, ip, port)
    except:
        print("Failed to receive data for SND_NKE")
        sock.close()
        sys.exit()
    
    if debug == DebugFlags.SendReceive or debug == DebugFlags.All:
        print("ACK")
        printhex(response)
    
    # Check for none ACK response.
    if len(response) != 1 or response[0] != 0xE5:
        print("Invalid response for SND_NKE")
        sys.exit()

# Send APP_RST and receive ACK.
if init == InitFlags.APP_RST:
    # Request format: 68 L L 68 C ADR CI CS 16
    start = b"\x68\x03\x03\x68"
    data = b"\x53" + struct.pack("B",int(adr)) + b"\x50"
    end = struct.pack("B",calcbcc(data)) + b"\x16"
    app_rst = start + data + end
    
    if debug == DebugFlags.SendReceive or debug == DebugFlags.All:
        print("APP_RST")
        printhex(app_rst)
    
    response = None
    try:
        response = SendAndReceive(app_rst, sock, protocol, 1, ip, port)
    except:
        print("Failed to receive data for APP_RST")
        sock.close()
        sys.exit()
    
    if debug == DebugFlags.SendReceive or debug == DebugFlags.All:
        print("ACK")
        printhex(response)
    
    # Check for none ACK response.
    if len(response) != 1 or response[0] != 0xE5:
        print("Invalid response for APP_RST")
        sys.exit()
    
dataRecord = 0          # Init data record index.
telegramCounter = 0     # Telegram counter.
primaryAddress = 0      # Primary address.
secondaryAddress = {}   # Secondary address.
run = True              # Keep on reading the M-Bus meter.
records = {}            # List of all records for a telegram.
recordsForTemplate = {} # List of records for template.

while run:

    # Send REQ_UD2 and receive RSP_UD.
    try:
        # Request format: 10 C ADR CS 16.
        c = toggle
        a = int(adr)
        cs = (c + a) & 0xFF
        req_ud2 = struct.pack("BBBBB", 0x10, c, a, cs, 0x16)
        
        if debug == DebugFlags.SendReceive or debug == DebugFlags.All:
            print("REQ_UD2")
            printhex(req_ud2)
        
        response = None
        try:
            response = SendAndReceive(req_ud2, sock, protocol, 1024, ip, port)
        except:
            print("Failed to receive data for REQ_UD2")
            sock.close()
            break
        
        if debug == DebugFlags.SendReceive or debug == DebugFlags.All:
            print("RSP_UD")
            printhex(response)

    except:
        print("Failed to read the M-Bus meter")
        break
    
    toggle = 0x5B if toggle == 0x7B else 0x7B
    telegramCounter += 1
    responseLength=len(response)
    
    # Check received bytes.
    if len(response) == 0:
        print("No response received")
        break
        
    # Check number of received bytes.
    if len(response) < 21:
        print("Too few bytes received")
        printhex(response)
        break
    
    # Check frame characters.
    if (response[0] != 0x68 or response[3] != 0x68):
        print("Incorrect frame characters")
        printhex(response)
        break
        
    # Check length. Allow for too long telegrams L = 06 means length of 268 bytes GG 2021-12-21
    if (response[1] != response[2] or (response[1] != responseLength-6 and response[1] != responseLength-262) ):
        print("Incorrect message length")
        printhex(response)
        break
        
    # Check control field.
    if response[4] != 0x08:
        print("Incorrect C (control) field")
        printhex(response)
        break

    # Check control information field.
    if response[6] != 0x72:
        print("Incorrect CI (control information) field")
        printhex(response)
        break
        
    # Check checksum.
    if (calcbcc(response[4:responseLength-2]) != response[responseLength-2]):
        print("Incorrect checksum")
        printhex(response)
        break

    primaryAddress = response[5]
    identification, manufacturer, version, medium, name, type = SecondaryAddress(response[7:15])
    secondaryAddress = (identification, manufacturer, version, medium, name, type)
    status = Status(response[16])

    print("")
    print(" --- Telegram %s ---"%(telegramCounter))
    print("")
    print("\tPrimary address:\t%s"%(primaryAddress))
    print("\tSecondary address:\t%s.%s.%s.%s [%s] [%s]"%(identification, manufacturer, version, medium, name, type))
    print("\tStatus:\t\t\t%02X %s"%(response[16], status))
    print("")

    # Print column header.
    if debug == DebugFlags.RecordData or debug == DebugFlags.All:
        print("{:3s} {:7s} {:40s} {:55s} {:s}".format("#", "Type", "Value", "Representation", "Data"))
        print("------------------------------------------------------------------------------------------------------------------------------------------------------------------------")
    else:
        print("{:3s} {:7s} {:40s} {:20s}".format("#", "Type", "Value", "Representation"))
        print("-----------------------------------------------------------------------------------------------------------")

    indexInTelegram = 19 
    record = (None, None, 0, "", False, False, None)  # Represents an object's entire record: (Value, Datatyp, length, Function, ManuData, MoreTelegrams, VIB).
    
    # ============================================================================
    # === Iterate over all objects in the telegram and display them in a table ===
    # ============================================================================ 
    while indexInTelegram + 2 < responseLength:
        dataRecord += 1
        try:
            record = decode_field(response[indexInTelegram:])  # Decode the object/record.
        except:
            record = ("", "", 1, "", False, False, "Error in datarecord.")
        
        # Result of record:
        value = record[0]
        dataType = record[1]
        length = record[2]
        function = record[3]
        manuData = record[4]
        moreTelegrams = record[5]
        vib = record[6]
        recordData = response[indexInTelegram:indexInTelegram+length]
        
        # Ignore Idle fillers.
        if dataType == "0x2F":
            indexInTelegram = indexInTelegram + length   # Point to next object in telegram.
            dataRecord -= 1
            continue
        
        records[dataRecord] = record    # Store record in list.
        
        # For each record, with debug flag for data, print: #, data type, value, representation and data.
        if debug == DebugFlags.RecordData or debug == DebugFlags.All:
            dataLine = DataLineAsHex(recordData)
            try:
                print("{:3s} {:7s} {:40s} {:55s} {:s}".format(str(dataRecord), str(dataType), str(value), vib, dataLine))
            except:
                # An exception might occur due to the normal string representation of the value failed.
                print("{:3s} {:7s} {:40s} {:55s} {:s}".format(str(dataRecord), str(dataType), "Unable to present the value", vib, dataLine))
                
                # The value's raw data bytes will be printed with an offset. The offset is only for aestetic reason.
                #data = "".join("%02X "%(i) for i in response[indexInTelegram:(indexInTelegram+length)])
                #PrintHexWithOffset(response[indexInTelegram:(indexInTelegram+length)], "            ")
                
        # For each record, print: #, data type, value and representation.
        else:
            # Print the record: record number, data type, value and representation.
            try:
                print("{:3s} {:7s} {:40s} {:20s}".format(str(dataRecord), str(dataType), str(value), vib))
            except:
                # An exception might occur due to the normal string representation of the value failed.
                print("{:3s} {:7s} {:40s} {:20s}".format(str(dataRecord), str(dataType), "Unable to present the value", vib))
                
                # The value's raw data bytes will be printed with an offset. The offset is only for aestetic reason.
                #data = "".join("%02X "%(i) for i in response[indexInTelegram:(indexInTelegram+length)])
                #PrintHexWithOffset(response[indexInTelegram:(indexInTelegram+length)], "            ")
            
        indexInTelegram = indexInTelegram + length   # Point to next object in telegram.
        
        # Manufacturer specify area present.
        if manuData:
            mfctlen  = responseLength - indexInTelegram - 2
            
            # Manufacturer specific data in telegram.
            if mfctlen  > 0:
                print("")
                print("Manufacturer specific data:")
                printhex(response[indexInTelegram-1:responseLength-2])  # Print MDH and bytes in manufacturer data.
                break
            # No manufacturer specific data in telegram.
            elif mfctlen == 0:
                print("")
                print("No bytes in manufacturer specific area.")
                print("")
                break
            # Flag in telegram for manufacturer specific data with negative manufacturer data length.
            else:
                print("")
                print("Error, length of manufacturer specific area < 0")
                print("")
        
    # Check that telegram length is invalid.
    if (responseLength - indexInTelegram < 2):
        print("")
        print("Error: Telegram length. Index is larger than telegram length.")
        print("")
    else:
        print("")
        print("Telegram Length is correct\n")
        print("")
        
    # ===============================================================
    # === Check if the data records should be saved as a template ===
    # ===============================================================
    while (False):
        key = input("Save records in template [y/n]? ")    # Get a single key press.

        # ASCII table: CR or 'y'.
        if len(key) == 0 or key[0] == 0x79:
            list = SelectRecordsForTemplate(records)
            recordsForTemplate.update(list)
            break
            
        # ASCII Table: ESC or 'n'.
        elif key[0] == 0x1B or key[0] == 'n':
            break
    
    records = {}
    
    # ======================================================
    # === If there is more telegrams to read, read next? ===
    # ======================================================
    if (moreTelegrams):
        while(True):
            key = input("Read next telegram [y/n]? ")    # Get a single key press.
            
            # ASCII table: CR or 'y'.
            if len(key) == 0 or key[0] == 'y':
                run = True
                print("")
                break
        
            # ASCII Table: ESC or 'n'.
            elif key[0] == 0x1B or key[0] == 'n':
                run = False
                break
    else:
        break
print("")

# Create a template of the selected records.
if len(recordsForTemplate) > 0:
    templateFolderPath = "%s%sPiiGAB%sBrowse%s"%(os.getenv("AppData"), os.sep, os.sep, os.sep)
    SetupTemplateFolder(templateFolderPath)
    templateFilePath = CreateTemplateFile(templateFolderPath, secondaryAddress[1], secondaryAddress[2], secondaryAddress[3])
    data = CreateTemplateData(recordsForTemplate, telegramCounter, primaryAddress, secondaryAddress)
    CreateTemplate(data, templateFilePath)
    print("Template file created at: %s"%(templateFilePath))
    print("")

sock.close()
