#!/usr/bin/python3

import serial,sys,time
import struct
import socket
import random
#import binascii
import string
import os
import subprocess

auchCRCHi = [ \
  0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,\
  0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0,\
  0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01,\
  0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41,\
  0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81,\
  0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0,\
  0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01,\
  0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,\
  0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,\
  0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0,\
  0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01,\
  0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,\
  0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,\
  0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0,\
  0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01,\
  0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,\
  0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,\
  0x40 ]

# Table of CRC values for loworder byte
# static char auchCRCLo[] = {
auchCRCLo = [ \
  0x00, 0xC0, 0xC1, 0x01, 0xC3, 0x03, 0x02, 0xC2, 0xC6, 0x06, 0x07, 0xC7, 0x05, 0xC5, 0xC4,\
  0x04, 0xCC, 0x0C, 0x0D, 0xCD, 0x0F, 0xCF, 0xCE, 0x0E, 0x0A, 0xCA, 0xCB, 0x0B, 0xC9, 0x09,\
  0x08, 0xC8, 0xD8, 0x18, 0x19, 0xD9, 0x1B, 0xDB, 0xDA, 0x1A, 0x1E, 0xDE, 0xDF, 0x1F, 0xDD,\
  0x1D, 0x1C, 0xDC, 0x14, 0xD4, 0xD5, 0x15, 0xD7, 0x17, 0x16, 0xD6, 0xD2, 0x12, 0x13, 0xD3,\
  0x11, 0xD1, 0xD0, 0x10, 0xF0, 0x30, 0x31, 0xF1, 0x33, 0xF3, 0xF2, 0x32, 0x36, 0xF6, 0xF7,\
  0x37, 0xF5, 0x35, 0x34, 0xF4, 0x3C, 0xFC, 0xFD, 0x3D, 0xFF, 0x3F, 0x3E, 0xFE, 0xFA, 0x3A,\
  0x3B, 0xFB, 0x39, 0xF9, 0xF8, 0x38, 0x28, 0xE8, 0xE9, 0x29, 0xEB, 0x2B, 0x2A, 0xEA, 0xEE,\
  0x2E, 0x2F, 0xEF, 0x2D, 0xED, 0xEC, 0x2C, 0xE4, 0x24, 0x25, 0xE5, 0x27, 0xE7, 0xE6, 0x26,\
  0x22, 0xE2, 0xE3, 0x23, 0xE1, 0x21, 0x20, 0xE0, 0xA0, 0x60, 0x61, 0xA1, 0x63, 0xA3, 0xA2,\
  0x62, 0x66, 0xA6, 0xA7, 0x67, 0xA5, 0x65, 0x64, 0xA4, 0x6C, 0xAC, 0xAD, 0x6D, 0xAF, 0x6F,\
  0x6E, 0xAE, 0xAA, 0x6A, 0x6B, 0xAB, 0x69, 0xA9, 0xA8, 0x68, 0x78, 0xB8, 0xB9, 0x79, 0xBB,\
  0x7B, 0x7A, 0xBA, 0xBE, 0x7E, 0x7F, 0xBF, 0x7D, 0xBD, 0xBC, 0x7C, 0xB4, 0x74, 0x75, 0xB5,\
  0x77, 0xB7, 0xB6, 0x76, 0x72, 0xB2, 0xB3, 0x73, 0xB1, 0x71, 0x70, 0xB0, 0x50, 0x90, 0x91,\
  0x51, 0x93, 0x53, 0x52, 0x92, 0x96, 0x56, 0x57, 0x97, 0x55, 0x95, 0x94, 0x54, 0x9C, 0x5C,\
  0x5D, 0x9D, 0x5F, 0x9F, 0x9E, 0x5E, 0x5A, 0x9A, 0x9B, 0x5B, 0x99, 0x59, 0x58, 0x98, 0x88,\
  0x48, 0x49, 0x89, 0x4B, 0x8B, 0x8A, 0x4A, 0x4E, 0x8E, 0x8F, 0x4F, 0x8D, 0x4D, 0x4C, 0x8C,\
  0x44, 0x84, 0x85, 0x45, 0x87, 0x47, 0x46, 0x86, 0x82, 0x42, 0x43, 0x83, 0x41, 0x81, 0x80,\
  0x40 ]


def CRC16(pszBuffer):
    uchCRCHi = 0xFF  # high byte of CRC initialized
    uchCRCLo = 0xFF  # low byte of CRC initialized
    for i in range(len(pszBuffer)):
        uIndex = uchCRCLo ^ pszBuffer[i] # calculate the CRC
        uchCRCLo = uchCRCHi ^ auchCRCHi[uIndex]
        uchCRCHi = auchCRCLo[uIndex] ;
    return (uchCRCLo << 8 | uchCRCHi) & 0xFFFF

def printhex(data):
    tstr = ""
    for j in range(len(data)):
        if ((j%16) == 0):
            tstr += "\n"
        tstr += "%02X "%data[j]
    print(tstr)



modbus_errors = {0x01:"Illegal Function Code",
                 0x02:"Illegal Data Address",
                 0x03:"Illegal Data Value",
                 0x04:"Server Failure",
                 0x05:"Acknowledge",
                 0x06:"Server Busy",
                 0x0A:"Gateway problem 0A",
                 0x0B:"Gateway problem 0B"}

class Source:
    def connect(self):
        raise NotImplementedError

    def write(self, data):
        raise NotImplementedError

    def read(self, length):
        raise NotImplementedError

    def close(self):
        raise NotImplementedError

class Ser(Source):
    def __init__(self):
        self.ser = None
        self.port = None
        self.bdrate = 9600
        self.timeout = 1.0
        self.bits = serial.EIGHTBITS
        self.parity = serial.PARITY_NONE
        self.stopb = serial.STOPBITS_ONE
        #self.other_port = other_port or port

    def connect(self, params = None):
        if 'port' in params:
            self.port = params['port']
        if 'baudrate' in params:
            self.bdrate = params['baudrate']
        if 'bytesize' in params:
            self.bits = params['bytesize']
        if 'parity' in params:
            self.parity = params['parity']
        if 'stopbits' in params:
            self.stopb = params['stopbits']
        if 'timeout' in params:
            self.timeout = params['timeout']
        try:
            self.ser = serial.Serial(self.port,#self.port,
                                 baudrate=self.bdrate,  # baudrate
                                 bytesize=self.bits,  # number of databits
                                 parity=self.parity,  # enable parity checking
                                 stopbits=self.stopb,  # number of stopbits
                                 timeout=self.timeout,  # set a timeout value, None for waiting forever
                                 xonxoff=0,  # enable software flow control
                                 rtscts=0  # enable RTS/CTS flow control
                                 )
        except:
            return False
        return True

    def write(self, data):
        self.p = 0
        self.arr = b""
        try:
            self.ser.write(data)
        except:
            return False
        return True

    def read(self, length):
        timeout = False
        ret = b""
        for i in range(length):
            try:
                a = self.ser.read(1)
            except:
                timeout = True
                break
            if a == b"":
                timeout = True
                break
            ret += a
        # if len(ret) != length:
        #    timeout = True #actually length is too short - but how to differentiate error messages?
        return (timeout, ret)
        # return ret

    def close(self):
        pass

class Tcp(Source):
    def __init__(self):
        self.sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.host = None
        self.port = None
        self.timeout = 1.0
        self.p = 0
        self.arr = b""

    def connect(self, params):
        if 'ipaddress' in params:
            self.host = params['ipaddress']
        if 'ipport' in params:
            self.port = params['ipport']
        if 'timeout' in params:
            self.timeout = params['timeout']
        retval = True
        try:
            self.sock.settimeout(self.timeout)
            self.sock.connect((self.host, self.port))
        except:
            retval = False
        return retval

    def write(self, data):
        self.p = 0
        self.arr = b""
        try:
            self.sock.sendall(data)
            print("write",("{:x} "*len(data)).format(*data))
        except :
            import traceback
            traceback.print_exc()
            return False
        return True

    def read(self, length):
        istimeout = False
        l = len(self.arr)
        if l < self.p + length:
            try:
                self.arr += self.sock.recv(1024)
                l = len(self.arr)
            except:
                istimeout = True
        if l >= self.p + length:
            a = self.arr[self.p:self.p + length]
            self.p += length
        else:
            a = self.arr[self.p:]
            self.p = len(self.arr)
            istimeout = True
        print("read", ("{:x} " * len(a)).format(*a))
        return (istimeout, a)

    def close(self):
        self.sock.close()


class Udp(Source):
    def __init__(self):
        # SOCK_DGRAM is the socket type to use for UDP sockets
        self.sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.host = None
        self.port = None
        self.timeout = 1.0
        self.p = 0
        self.arr = b""

    def connect(self, params):
        if 'ipaddress' in params:
            self.host = params['ipaddress']
        if 'ipport' in params:
            self.port = params['ipport']
        if 'timeout' in params:
            self.timeout = params['timeout']
        self.sock.settimeout(self.timeout)
        return True

    def write(self, data):
        self.p = 0
        self.arr = b""
        try:
            self.sock.sendto(data, (self.host, self.port))
        except:
            return False
        return True

    def read(self, length):
        istimeout = False
        l = len(self.arr)
        if l < self.p + length:
            try:
                self.arr += self.sock.recv(1024)
                l = len(self.arr)
            except:
                istimeout = True
        if l >= self.p + length:
            a = self.arr[self.p:self.p + length]
            self.p += length
        else:
            a = self.arr[self.p:]
            self.p = len(self.arr)
            istimeout = True
        return (istimeout, a)

    def close(self):
        pass



class modbus(Ser, Tcp, Udp):
    def __init__(self, porttype,  modbustype, params,  addr ):
        self.params = params
        if porttype == 'SER':
            self.prt = Ser()
        elif porttype == 'TCP':
            self.prt = Tcp()
        else: # porttype == 'UDP':
            self.prt = Udp()
        if modbustype == "RTU":
            self.mtype = "RTU"
            self.hdr = 0
        else:
            self.mtype = "TCP"
            self.hdr = 6
        self.address = addr
        self.status = False
        self.arr = b""
        self.request = b""
        self.nregs = 0

    def reset(self):
        self.arr = b""

    def readFromDevice(self, fcode,  streg,  nregs):
        subprocess.call(["killall","modbus2mbus"])
        #Create request
        error_msg = ""
        if not self.prt.connect(self.params):
            return (False, "Connection failed.")
        to = False
        self.request = struct.pack(">BBHH", self.address, fcode, streg, nregs)
        if self.mtype=="RTU":
            crc = CRC16(self.request)
            self.request += struct.pack(">H",crc)
        else:
            tid = int(random.random()*100000)&0xFFFF
            olen = len(self.request)
            self.request = struct.pack(">HHH", tid, 0, olen) + self.request
        if not self.prt.write(self.request):
            return (False,"Write error")
        #printhex(self.request)
        time.sleep(0.1)
        self.arr = b""
        if self.mtype=="TCP":
            to, a = self.prt.read(6) # TCP header
            self.arr=a
            if to:
                error_msg="Timeout"
                self.prt.close()
                return (False,error_msg)
            print("Len Arr = %d"%len(self.arr))
            intid = struct.unpack(">H", self.arr[0:2])[0]
            if (intid != tid):
                error_msg="TID's don't match"
                self.prt.close()
                return (False,error_msg)
        to, a = self.prt.read(3)
        self.arr +=a
        if to:
            error_msg="Timeout"
            return (False, error_msg)
        raddr = self.arr[self.hdr+0] # Byte 0, function code
        if raddr != self.address:
            error_msg="Answer from a wrong unit, %d"%raddr
            return (False,error_msg)
        rfunccode  = self.arr[self.hdr+1] # Byte 1, function code
        if rfunccode & 0x80:
            e = self.arr[self.hdr+2] # Byte 2, exception
            if e > 6:
                error_msg="Error code %d is out of bounds"%e
            else:
                error_msg=modbus_errors[e]
            return (False,error_msg)
        else:
            rbcnt = self.arr[self.hdr+2] # Byte 2, byte count
            if rbcnt != 2*nregs:
                error_msg="Byte count error, byte count = %d, nregs = %d"%(rbcnt, nregs)
                return (False,error_msg)
            for i in range(rbcnt//2):
                to, a = self.prt.read(2)
                self.arr += a
                if to:
                    self.prt.close()
                    return (False,"Timeout")
            if self.mtype == "RTU":
                to, a = self.prt.read(2)
                self.arr += a
                if to:
                    self.prt.close()
                    return (False,"Timeout")
                r = len(self.arr)
                calcCRC = CRC16(self.arr[0:r-2])
                recCRC = -1
                try:
                    recCRC = struct.unpack(">H",self.arr[r-2:r])[0]
                except:
                    pass
                if calcCRC != recCRC:
                    error_msg = "Checksum NOT OK %X %X"%(calcCRC,recCRC)
                    return (False,error_msg)
        self.nregs = nregs
        return (True,"")

    def getArray(self, streg, nregs):
        arr = b""
        if len(self.arr) < self.hdr+3+2*nregs:
            return (False, "")
        rbcnt = self.arr[self.hdr+2] # Byte 2, byte count
        if rbcnt < 2*nregs:
            return (False, "")
        arr = self.arr[self.hdr+3:self.hdr+3+2*nregs]
        return (True, arr) #Contains requested registers

    def rearrangeBytes(self, inarr, byteorder) :
        arr = b""
        l = len(inarr)
        if (l%2) != 0:
            return (False, "")
        if (byteorder == "big"):
            arr = inarr
        elif (byteorder == "little"):
            for i in range(l):
                arr += inarr[l-i-1:l-i]
        elif (byteorder == "byteswap"):
            for i in range(l//2):
                arr += inarr[l - 2*i - 2:l - 2*i - 1]
                arr += inarr[l - 2*i - 1:l - 2*i ]
        elif (byteorder == "regswap"):
            for i in range(l//2):
                arr += inarr[2*i+1:2*i+2]
                arr += inarr[2*i:2*i+1]
        return (True, arr)

    def arrayToValue(self, inarr,  valuetype):
        val = ""
        l = len(inarr)
        if (l&1) != 0:
             return (False, )
        if (l == 2):
            if valuetype == "float":
                val = "NAN"
            elif valuetype == "int":
                val = "%d"%struct.unpack(">h", inarr)[0]
            elif valuetype == "uint":
                val = "%d"%struct.unpack(">H", inarr)[0]
        elif (l == 4):
            if valuetype == "float":
                val = "%f"%struct.unpack(">f", inarr)[0]
            elif valuetype == "int":
                val = "%d"%struct.unpack(">i", inarr)[0]
            elif valuetype == "uint":
                val = "%d"%struct.unpack(">I", inarr)[0]
        elif (l ==8):
            if valuetype == "float":
                val = "%f"%struct.unpack(">d", inarr)[0]
            elif valuetype == "int":
                val = "%d"%struct.unpack(">q", inarr)[0]
            elif valuetype == "uint":
                val = "%d"%struct.unpack(">Q", inarr)[0]
        else:
            val = "Not impl"
        return val
    
    def getValue(self, streg, nregs, valuetype, byteorder):
        tf,  arr = self.getArray( streg, nregs)
        if tf == False:
            return (False, )
        tf, arr = self.rearrangeBytes(arr, byteorder) 
        if valuetype == "hex":
            s = ""
            for i in arr:
                s += "%02X "%i
            return (tf,  s)
        if tf == False:
            return (False, )
        val = self.arrayToValue(arr, valuetype)
        return (tf, val)
    
    def getString(self, streg, nregs):
        s = ""
        tf, arr = self.getArray(streg, nregs)
        if tf:
            for c in arr:
                #if "%c"%c in string.ascii_letters+string.digits:
                if c >= 0x20 and c < 0x7F:
                    s += "%c "%c
                else:
                    s += "%02Xh "%c
        return s

    def getResponse(self):
        s = ""
        for c in self.arr:
            s += "%02X "%c
        return s
    
    def getRequest(self):
        s = ""
        for c in self.request:
            s += "%02X "%c
        return s

if __name__=="__main__":
    if len(sys.argv) < 7:
        print("Usage: %s <address> <function> <starting address> <# registers> <serial/tcp/udp> <RTU/TCP> "%sys.argv[0])
        sys.exit(0)
    params = {}
    params['port'] = "/dev/ttyUSB0"
    params['baudrate'] = 9600
    params['parity'] = serial.PARITY_NONE
    params['ipaddress'] = "10.66.57.93"
    params['ipport'] = 10002
    adress = int(sys.argv[1])
    fcode  = int(sys.argv[2])
    streg = int(sys.argv[3])
    nregs  = int(sys.argv[4])
    devicetype   = sys.argv[5]
    modbustype = sys.argv[6]
    reglst = []
    mbs = modbus(devicetype, modbustype, params, adress)
    tf, error_msg = mbs.readFromDevice(fcode, streg, nregs)
    if (tf):
        tf, val = mbs.getValue(streg, nregs, "float","big")
        print("BigEnd FloatVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "int","big")
        print("BigEnd IntVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "uint","big")
        print("BigEnd UIntVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "hex","big")
        print("Hex big = %s"%val )
        
        tf, val = mbs.getValue(streg, nregs, "float","little")
        print("LittleEnd FloatVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "int","little")
        print("LittleEnd IntVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "uint","little")
        print("LittleEnd UIntVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "hex","little")
        print("Hex little = %s"%val )

        tf, val = mbs.getValue(streg, nregs, "float","regswap")
        print("Regswap FloatVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "int","regswap")
        print("Regswap IntVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "uint","regswap")
        print("Regswap UIntVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "hex","regswap")
        print("Hex regswap = %s"%val )

        tf, val = mbs.getValue(streg, nregs, "float","byteswap")
        print("Byteswap FloatVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "int","byteswap")
        print("Byteswap IntVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "uint","byteswap")
        print("Byteswap UIntVal = %s"%val )
        tf, val = mbs.getValue(streg, nregs, "hex","byteswap")
        print("Hex byteswap = %s"%val )
        
        print("string = %s"%mbs.getString(streg, nregs) )       
    else:
        print(mbs.getRequest())
        print(mbs.getResponse())
        print("Error: "+error_msg)

